from tractor.secrets import Secrets
from tractor_disk.settings import settings
from tractor.util.retrying import retry
import requests
import uuid
import xmltodict


class MDSHttpError(Exception):
    def __init__(self, status_code, message):
        self.status_code = status_code
        self.message = message

    def __repr__(self):
        return "<Status code: {}. Details: {}>".format(self.status_code, self.message)

    __str__ = __repr__


def execute_and_rewrap(f, *args):
    try:
        return f(*args)
    except requests.HTTPError as e:
        raise MDSHttpError(e.response.status_code, str(e))


class MDSClient:
    def __init__(self, read_url: str, write_url: str, namespace: str, expire: str = None):
        self.read_url = read_url
        self.write_url = write_url
        self.namespace = namespace
        self.expire = expire
        self.settings = settings().mds

    def _upload_url(self, filename: str) -> str:
        return f"{self.write_url}/upload-{self.namespace}/{filename}.{uuid.uuid4().hex}"

    def _download_url(self, filepath: str) -> str:
        return f"{self.read_url}/get-{self.namespace}/{filepath}"

    def upload(self, filename: str, data: bytes) -> str:
        @retry(
            base_delay=self.settings.retrying.base_delay_in_seconds,
            retries=self.settings.retrying.count,
        )
        def upload_impl(self, filename: str, data: bytes) -> str:
            url = self._upload_url(filename)
            params = {"expire": self.expire} if self.expire else None
            headers = {
                "X-Ya-Service-Ticket": Secrets().mds_secret(),
            }
            resp = requests.post(url, params=params, headers=headers, data=data)
            resp.raise_for_status()
            resp_data = xmltodict.parse(resp.content, dict_constructor=dict)
            return resp_data["post"]["@key"]

        return execute_and_rewrap(upload_impl, self, filename, data)

    def download(self, filepath: str) -> bytes:
        @retry(
            base_delay=self.settings.retrying.base_delay_in_seconds,
            retries=self.settings.retrying.count,
        )
        def download_impl(self, filepath: str) -> bytes:
            url = self._download_url(filepath)
            headers = {
                "X-Ya-Service-Ticket": Secrets().mds_secret(),
            }
            resp = requests.get(url, headers=headers)
            resp.raise_for_status()
            return resp.content

        return execute_and_rewrap(download_impl, self, filepath)


def create_mds_client(
    read_url: str, write_url: str, namespace: str, expire: str = None
) -> MDSClient:
    return MDSClient(read_url, write_url, namespace, expire)


def create_mds_client_from_settings() -> MDSClient:
    mds_settings = settings().mds
    return create_mds_client(
        mds_settings.read_url, mds_settings.write_url, mds_settings.namespace, mds_settings.expire
    )
