from typing import Any, Dict
import json
import io
from urllib.parse import urlencode

from googleapiclient.http import MediaIoBaseDownload, HttpRequest

from tractor.util.retrying import need_retry, TemporaryError, retry
from tractor_disk.ms_auth_http import Credentials, AuthorizedHttp
from tractor_disk.common import LOST_AND_FOUND_DIR, replace_slash
from tractor_disk.ms_file import MS_FOLDER_MIME_TYPE_PLACEHOLDER
from tractor_disk.disk_error import DiskTemporaryError, DiskPermanentError, catch_errors
from tractor_disk.settings import settings

FIELDS_LISTING = ["parentReference", "id", "name", "file", "size", "folder", "root"]


DISK_NAME = "OneDrive"


def catch_disk_errors(method):
    return catch_errors(method, _dispatch_disk_error)


def _dispatch_disk_error(exc: Exception):
    try:
        raise exc
    except (DiskTemporaryError, DiskPermanentError) as disk_err:
        if not disk_err.source:
            disk_err.source = DISK_NAME
        raise disk_err
    except (TemporaryError, TimeoutError, ConnectionError, OSError) as tmp_err:
        raise DiskTemporaryError(str(tmp_err), DISK_NAME)


class OneDrive:
    def __init__(self, secret, user_email, chunk_size=10 * 1024 * 1024):
        domain = user_email[user_email.rfind("@") + 1 :]
        self.http = AuthorizedHttp(Credentials(secret, domain))
        self.user_email = user_email
        self.root_id = None
        self.files = None
        self.chunk_size = chunk_size
        self.settings = settings().ms_drive

    def root_folder_id(self):
        if self.root_id:
            return self.root_id
        resp = self._api_get(resource="root")
        self.root_id = resp.get("id")
        if not self.root_id:
            raise RuntimeError("Cannot get root folder id")
        return self.root_id

    def get_files(self):
        if self.files:
            return self.files
        self.files = []
        self._fill_files(self.root_folder_id())
        return self.files

    @staticmethod
    def is_downloadable(file_metadata: Dict[str, Any]) -> bool:
        return True

    def _fill_files(self, item_id):
        children = self._api_get_paged(
            item_id=item_id, resource="children", params={"$select": ",".join(FIELDS_LISTING)}
        )
        for child in children:
            self.files.append(child)
            if "folder" in child:
                self._fill_files(child["id"])

    @catch_disk_errors
    def download_file(self, file_id):
        url = self._url(item_id=file_id, resource="content")
        request = HttpRequest(self.http, lambda *args: None, url)
        content = io.BytesIO()
        downloader = MediaIoBaseDownload(content, request, chunksize=self.chunk_size)
        done = False
        while not done:
            status, done = downloader.next_chunk(num_retries=self.settings.retrying.count)
            chunk = content.getvalue()
            # print ("Download {}%, chunk size={} done={}".format(int(status.progress() * 100), len(chunk), done))
            yield chunk
            content.truncate(0)
            content.seek(0)

    def folder_mime_type(self):
        return MS_FOLDER_MIME_TYPE_PLACEHOLDER

    @catch_disk_errors
    def _api_get(self, item_id=None, resource=None, params=None):
        url = self._url(item_id, resource)
        return _request(self.http, url, self.settings.retrying, params)

    @catch_disk_errors
    def _api_get_paged(self, item_id=None, resource=None, params=None):
        url = self._url(item_id, resource)
        return _request_multiple_pages(self.http, url, self.settings.retrying, params)

    def _url(self, item_id=None, resource=None):
        url = "https://graph.microsoft.com/v1.0/users/{}/drive".format(self.user_email)
        if item_id:
            url += "/items/{}".format(item_id)
        if resource:
            url += "/{}".format(resource)
        return url


class OneDriveBuildPathMappingOp:
    def __init__(self, files):
        self.files = files

    def __call__(self):
        path_by_id_mapping = {}
        for file in self.files:
            path_by_id_mapping[file["id"]] = _file_path(file)
        return path_by_id_mapping, {}


def _file_path(file):
    if "root" in file:
        return "/"
    PATH_PREFIX = "/drive/root:"
    parent_ref = file.get("parentReference")
    if parent_ref and "path" in parent_ref:
        folder = parent_ref["path"][len(PATH_PREFIX) :]
    else:
        folder = LOST_AND_FOUND_DIR
    return folder + "/" + replace_slash(file["name"])


def _request_multiple_pages(http, url, retry_policy, params=None):
    res = []
    while True:
        resp = _request(http, url, retry_policy, params)
        res += resp.get("value", [])
        url = resp.get("@odata.nextLink")
        params = {}
        if not url:
            break
    return res


def _request(http, url, retry_policy, params=None):
    uri = url
    if params:
        uri += "?{}".format(urlencode(params))

    @retry(base_delay=retry_policy.base_delay_in_seconds, retries=retry_policy.count)
    @catch_disk_errors
    def impl():
        response, content = http.request(uri)
        if response.status == 200:
            return json.loads(content)
        msg = '{} {}. Details: "{}"'.format(response.status, response.reason, str(content))
        if need_retry(response.status):
            raise DiskTemporaryError(msg, DISK_NAME)
        raise DiskPermanentError(msg, DISK_NAME)

    return impl()
