from dataclasses import dataclass
from tractor.util.dataclasses import construct_from_dict
from tractor.settings import (
    BlackboxSettings,
    DirectorySettings,
    RetryingSettings,
    RetryingSettingsWithDelay,
    TractorDBSettings,
    Settings as BaseSettings,
)


@dataclass
class YandexDiskSettings:
    host: str
    user_agent: str = "tractor_disk"
    verify_ssl_on_upload: bool = True
    retrying: RetryingSettingsWithDelay = RetryingSettingsWithDelay(
        base_delay_in_seconds=0.100, count=2
    )


@dataclass
class GoogleDriveSettings:
    retrying: RetryingSettings = RetryingSettings(count=3)


@dataclass
class MsDriveSettings:
    retrying: RetryingSettingsWithDelay = RetryingSettingsWithDelay(
        base_delay_in_seconds=0.100, count=3
    )


@dataclass
class SyncSettings:
    validate_domain: bool = True
    worker_sleep_period_in_seconds: int = 60
    worker_chunk_size: int = 1000


@dataclass
class ListSettings:
    worker_sleep_period_in_seconds: int = 60


@dataclass
class MDSSettings:
    read_url: str
    write_url: str
    namespace: str
    retrying: RetryingSettingsWithDelay
    expire: str = None


@dataclass
class TaskingSettings:
    expiry_timeout_in_seconds: int = 10 * 60
    heartbeat_rest_period_in_seconds: int = 5 * 60

    def __post_init__(self):
        if self.heartbeat_rest_period_in_seconds >= self.expiry_timeout_in_seconds:
            # The whole heartbeat period would be even bigger.
            raise ValueError("Heartbeat period would, but should not exceed expiry timeout.")


@dataclass
class LoggingSettings:
    file: str = None


@dataclass
class CoordinatorSettings:
    sleep_period_in_seconds: int = 1
    sync_migrations_pagination_size: int = 100


class Settings(BaseSettings):
    def __init__(self) -> None:
        super().__init__()
        self.directory: DirectorySettings = construct_from_dict(
            DirectorySettings, self.data.get("directory")
        )
        self.blackbox: BlackboxSettings = construct_from_dict(
            BlackboxSettings, self.data.get("blackbox")
        )
        self.yandex_disk: YandexDiskSettings = construct_from_dict(
            YandexDiskSettings, self.data.get("yandex_disk")
        )
        self.google_drive: GoogleDriveSettings = construct_from_dict(
            GoogleDriveSettings, self.data.get("google_drive")
        )
        self.ms_drive: MsDriveSettings = construct_from_dict(
            MsDriveSettings, self.data.get("ms_drive")
        )
        self.sync: SyncSettings = construct_from_dict(SyncSettings, self.data.get("sync"))
        self.list: ListSettings = construct_from_dict(ListSettings, self.data.get("list"))
        self.tractor_disk_db: TractorDBSettings = construct_from_dict(
            TractorDBSettings, self.data.get("tractor_db")
        )
        self.mds: MDSSettings = construct_from_dict(MDSSettings, self.data.get("mds"))
        self.tasking: TaskingSettings = construct_from_dict(
            TaskingSettings, self.data.get("tasking")
        )
        self.logging: LoggingSettings = construct_from_dict(
            LoggingSettings, self.data.get("logging", {})
        )
        self.coordinator: CoordinatorSettings = construct_from_dict(
            CoordinatorSettings, self.data.get("coordinator", {})
        )


def settings() -> Settings:
    if not hasattr(settings, "instance"):
        settings.instance = Settings()

    return settings.instance
