from enum import Enum
from functools import partial

from tractor_disk.google_drive import GoogleDrive, GoogleBuildPathMappingOp
from tractor_disk.google_file import GoogleFile
from tractor_disk.ms_drive import OneDrive, OneDriveBuildPathMappingOp
from tractor_disk.ms_file import OneDriveFile
from tractor.secrets import Secrets


class SourceDrive(Enum):
    GOOGLE = "google"
    MICROSOFT = "ms"

    @staticmethod
    def from_string(s):
        try:
            return SourceDrive(s)
        except KeyError:
            raise ValueError("Invalid source drive")

    def fake_src_login(self):
        if self == SourceDrive.GOOGLE:
            return "migration-meta"
        if self == SourceDrive.MICROSOFT:
            return "migration-meta-ms"


def get_drive_by_source(source: SourceDrive):
    if source == SourceDrive.GOOGLE:
        return GoogleDrive
    if source == SourceDrive.MICROSOFT:
        return OneDrive
    raise RuntimeError("Invalid source drive type: {}".format(source))


def get_path_mapping_op_by_source(source: SourceDrive):
    if source == SourceDrive.GOOGLE:
        return GoogleBuildPathMappingOp
    if source == SourceDrive.MICROSOFT:
        return OneDriveBuildPathMappingOp
    raise RuntimeError("Invalid source drive type: {}".format(source))


def get_file_by_source(source: SourceDrive):
    if source == SourceDrive.GOOGLE:
        return GoogleFile
    if source == SourceDrive.MICROSOFT:
        return OneDriveFile
    raise RuntimeError("Invalid source drive type: {}".format(source))


def get_create_drive_op(source: SourceDrive):
    if source == SourceDrive.GOOGLE:
        return create_google_drive
    if source == SourceDrive.MICROSOFT:
        return create_ms_drive
    raise RuntimeError("Invalid source drive type: {}".format(source))


def create_google_drive(email):
    secrets = Secrets()
    return GoogleDrive(secrets.google_secret(), email)


def create_ms_drive(email):
    secrets = Secrets()
    return OneDrive(secrets.ms_secret(), email)


def get_create_drive_op_with_explicit_secret(secret, source: SourceDrive):
    if source == SourceDrive.GOOGLE:
        return partial(create_google_drive_with_explicit_secret, secret)
    if source == SourceDrive.MICROSOFT:
        return partial(create_ms_drive_with_explicit_secret, secret)
    raise RuntimeError("Invalid source drive type: {}".format(source))


def create_google_drive_with_explicit_secret(secret: object, email):
    return GoogleDrive(secret, email)


def create_ms_drive_with_explicit_secret(secret: object, email):
    return OneDrive(secret, email)


def get_drive_secret(source):
    if source == SourceDrive.GOOGLE:
        return Secrets().google_secret()
    if source == SourceDrive.MICROSOFT:
        return Secrets().ms_secret()
