from .sync_file import SyncFileOp
from tractor_disk.disk_pair import DiskPair
from tractor.logger import get_logger
import json
import traceback
from itertools import islice


class SyncOrgOp:
    def __init__(self, create_dst_disk, create_src_disk, worker_num=0, workers_count=1):
        self.create_dst_disk = create_dst_disk
        self.create_src_disk = create_src_disk
        self.worker_num = worker_num
        self.workers_count = workers_count
        self.logger = get_logger()

    def __call__(self, import_file_path):
        import_data = self._load_import_data(import_file_path)
        self.logger.info("sync started")
        row_current = 0
        rows_total = len(import_data) / self.workers_count
        total_synced = 0
        total_exists = 0
        total_errors = 0
        for row in islice(import_data, self.worker_num, None, self.workers_count):
            user, file = self._parse_row(row)
            if user["uid"] == "NULL":
                continue
            try:
                disk_pair = DiskPair(user, self.create_src_disk, self.create_dst_disk)
                op = SyncFileOp(disk_pair)
                synced = op(file)
                if synced:
                    total_synced += 1
                    self.logger.info(
                        "file synced",
                        org_id=user["org_id"],
                        uid=user["uid"],
                        file_id=file["id"],
                        mime_type=file["mimeType"],
                        login=user["login"],
                    )
                else:
                    total_exists += 1
            except Exception as e:
                total_errors += 1
                stack_summary = traceback.extract_tb(e.__traceback__)
                last_frame = stack_summary[-1]
                last_frame = str(f"{last_frame.filename}:{last_frame.lineno} in {last_frame.name}")
                self.logger.error(
                    "sync file error",
                    org_id=user["org_id"],
                    uid=user["uid"],
                    file_id=file["id"],
                    reason=str(e),
                    last_frame=last_frame,
                    mime_type=file["mimeType"],
                    login=user["login"],
                )

            if row_current % 100 == 0:
                progress = 100 * (row_current / rows_total if rows_total > 0 else 1)
                self.logger.info(
                    "status",
                    org_id=user["org_id"],
                    progress=progress,
                    total_synced=total_synced,
                    total_exists=total_exists,
                    total_errors=total_errors,
                )
            row_current += 1

        self.logger.info("sync finished")

    def _load_import_data(self, import_file_path):
        with open(import_file_path, "r") as f:
            return json.load(f)

    def _parse_row(self, row):
        user = {
            "org_id": row["orgid"],
            "domain": row["domain"],
            "login": row["user_login"],
            "email": "{}@{}".format(row["user_login"], row["domain"]),
            "uid": str(row["yandex_puid"]),
        }
        file = {"id": row["file_id"], "path": row["file_path"], "mimeType": row["file_mime_type"]}
        return user, file
