#!/usr/bin/env python3

from tractor_disk.yandex_disk import create_yandex_disk
from tractor_disk.google_drive import FOLDER_MIME_TYPE as GOOGLE_FOLDER_MIME_TYPE
from tractor_disk.ms_file import MS_FOLDER_MIME_TYPE_PLACEHOLDER as MS_FOLDER_MIME_TYPE

import os
import json
import argparse
from humanize import naturalsize

PREFIX = "disk:"
FOLDER_MIME_TYPES = (GOOGLE_FOLDER_MIME_TYPE, MS_FOLDER_MIME_TYPE)


def main():
    args = parse_args()

    with open(args.report, "r") as file:
        report = json.load(file)

    uid = str(report[0].get("yandex_puid"))
    source_files = get_source_disk_files(report)
    ya_disk_files = get_yandex_disk_files(uid)

    if args.dir:
        os.makedirs(args.dir, exist_ok=True)
        report_filename = os.path.basename(args.report)
        save_files(sorted(source_files), os.path.join(args.dir, "src"))
        save_files(sorted(ya_disk_files), os.path.join(args.dir, "dst"))

    (diff_left, diff_right) = diff(source_files, ya_disk_files)

    if args.summary:
        src_len = len(source_files)
        dst_len = len(ya_disk_files)
        src_size = naturalsize(sum([int_or_zero(f[1]) for f in source_files]), gnu=True)
        dst_size = naturalsize(sum([int_or_zero(f[1]) for f in ya_disk_files]), gnu=True)
        dleft = len(diff_left)
        dright = len(diff_right)
        print(
            f"src-len:{src_len} dst-len:{dst_len} src-size:{src_size} dst-size:{dst_size} diff-left:{dleft} diff-right:{dright}"
        )
        return

    for file in diff_left:
        print(f"< {file}")
    for file in diff_right:
        print(f"> {file}")


def parse_args():
    parser = argparse.ArgumentParser(
        description="List user yandex disk and build list of not synced files"
    )
    parser.add_argument("-r", "--report", help="user report.json file", required=True)
    parser.add_argument("-d", "--dir", help="output directory for files with paths")
    parser.add_argument("-s", "--summary", help="print short summary", action="store_true")
    args = parser.parse_args()
    return args


def get_source_disk_files(report):
    files = filter(lambda f: f.get("file_mime_type") not in FOLDER_MIME_TYPES, report)
    return [(unify_path(file["file_path"]), file["file_size"]) for file in files]


def get_yandex_disk_files(uid):
    disk = create_yandex_disk(uid)
    files = disk.files()
    return [(unify_path(file.get("path")[len(PREFIX) :]), file.get("size")) for file in files]


def diff(source_files, dest_files):
    src = set([x[0] for x in source_files])
    dst = set([x[0] for x in dest_files])
    return (src - dst, dst - src)


def save_files(files, path):
    with open(path, "w") as f:
        for file in files:
            f.write(f"{file[0]}\t{file[1]}\n")


def unify_path(path):
    return path if path[0] == "/" else "/" + path


def int_or_zero(value):
    return int(value if value != "NULL" else 0)


if __name__ == "__main__":
    main()
