#!/usr/bin/env python3

from tractor_disk.yandex_disk import create_yandex_disk
import sys
import json
import argparse
from humanize import naturalsize


def main(args):
    plan = json.loads(sys.stdin.read())

    users = {}
    user_files_sz = {}
    user_files_count = {}
    total_files_sz = 0
    total_files_count = 0

    for item in plan:
        user_login = item["user_login"]
        user_uid = item["yandex_puid"]
        users[user_login] = user_uid
        user_files_sz[user_login] = user_files_sz.get(user_login, 0) + int_or_zero(
            item["file_size"]
        )
        user_files_count[user_login] = user_files_count.get(user_login, 0) + 1
        total_files_sz += int_or_zero(item["file_size"])
        total_files_count += 1

    for user, uid in users.items():
        disk = create_yandex_disk(str(uid))
        try:
            quote = disk.quote()
            used_space = quote.get("used_space", 0)
            total_space = quote.get("total_space", 0)
            quota_status = "OK" if user_files_sz[user] < total_space else "EXCEEDED"
        except:
            used_space = 0
            total_space = 0
            quota_status = "UNKNOWN"
        diff = round(user_files_sz[user] / used_space, 2) if used_space > 0 else "-"
        print(
            user,
            user_files_count[user],
            num_or_humanize(user_files_sz[user], args.humanize),
            num_or_humanize(used_space, args.humanize),
            num_or_humanize(total_space, args.humanize),
            quota_status,
            diff,
        )
    print(
        "total users:{} files:{} size:{}".format(
            len(users), total_files_count, num_or_humanize(total_files_sz, args.humanize)
        ),
        file=sys.stderr,
    )


def num_or_humanize(value, humanize):
    return naturalsize(value, gnu=True) if humanize else value


def int_or_zero(value):
    return int(value if value != "NULL" else 0)


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument("--humanize", action="store_true", help="print human readably size values")
    args = parser.parse_args()
    return args


if __name__ == "__main__":
    main(parse_args())
