import copy
import json
from typing import Dict

from tractor.disk.db import Database
from tractor.error import *
from tractor.models import (
    Task,
    TaskWorkerStatus,
)
from tractor.disk.models import (
    TaskType,
    UserMigration,
    UserMigrationStatus,
    UserStatistics,
)

from tractor.logger import DeployLogger
from tractor_disk.workers.env import Env


def processes_finished_list_task(migration: UserMigration, task: Task, env: Env, cur):
    logger: DeployLogger = env["logger"]
    db: Database = env["db"]
    if task.canceled or migration.status == UserMigrationStatus.CANCELING:
        migration.status = UserMigrationStatus.ERROR
        migration.error_reason = OPERATION_CANCELED
    elif task.worker_status != TaskWorkerStatus.SUCCESS:
        migration.status = UserMigrationStatus.ERROR
        migration.error_reason = _error_reason(task)
    # sanity check
    elif migration.status != UserMigrationStatus.LISTING:
        migration.status = UserMigrationStatus.ERROR
        migration.error_reason = WRONG_STATE
        pass

    _fill_statistics_from_list_worker_output(migration.stats, task.worker_output)

    if migration.status == UserMigrationStatus.ERROR:
        logger.info(
            message="update user migration",
            new_status=migration.status,
            error_reason=migration.error_reason,
            org_id=migration.org_id,
            task_id=migration.list_task_id,
            login=migration.login,
        )
        db.mark_migration_finished(
            org_id=migration.org_id,
            login=migration.login,
            status=migration.status.value,
            error_reason=migration.error_reason,
            stats=migration.stats,
            cur=cur,
        )
        return

    files_count = task.worker_output["files_count"]

    if files_count == 0:
        migration.status = UserMigrationStatus.SUCCESS
        logger.info(message="update user migration", new_status=migration.status)
        db.mark_migration_finished(
            org_id=migration.org_id,
            login=migration.login,
            status=migration.status.value,
            error_reason="",
            stats=migration.stats,
            cur=cur,
        )
        return

    split_by = env["settings"].sync.worker_chunk_size
    workers_count = files_count // split_by
    if (files_count % split_by) != 0:
        workers_count += 1

    worker_input = copy.deepcopy(task.input)
    worker_input["workers_count"] = workers_count
    worker_input["stid"] = task.worker_output["stid"]

    for worker_num in range(workers_count):
        worker_input["worker_num"] = worker_num
        inp = json.dumps(worker_input)
        sync_id = db.create_task(
            type=TaskType.SYNC, org_id=task.org_id, domain=task.domain, worker_input=inp, cur=cur
        )

        migration.sync_task_ids.append(sync_id)

    migration.status = UserMigrationStatus.SYNCING
    logger.info(message="update user migration", new_status=migration.status)
    db.update_migration_to_sync_state(
        org_id=migration.org_id,
        login=migration.login,
        sync_task_ids=migration.sync_task_ids,
        stats=migration.stats,
        cur=cur,
    )


def _error_reason(task):
    if "error" not in task.worker_output:
        return LISTING_ERROR
    if task.worker_output["error"] == NOT_ENOUGH_QUOTA:
        return NOT_ENOUGH_QUOTA
    if task.worker_output["error"] == YANDEX_USER_NOT_FOUND:
        return YANDEX_USER_NOT_FOUND
    if task.worker_output["error"] == EXTERNAL_USER_NOT_FOUND:
        return EXTERNAL_USER_NOT_FOUND
    return LISTING_ERROR


def _fill_statistics_from_list_worker_output(s: UserStatistics, d: Dict):
    if "quota" in d and isinstance(d["quota"], int):
        s.quota = d["quota"]
    if "files_size" in d and isinstance(d["files_size"], int):
        s.files_size = d["files_size"]
    if "files_count" in d and isinstance(d["files_count"], int):
        s.files_count = d["files_count"]
