from contextlib import contextmanager, closing
from threading import Event, Thread

import psycopg2
import psycopg2.extensions

from tractor.disk.db import Database
from tractor_disk.settings import Settings
from tractor_disk.workers.env import Env


if psycopg2.threadsafety < 1:
    raise Exception("`psycopg2.threadsafety` is needed to be at least 1 (module shareability).")


def _run(env: Env, task_id: int, cursor: psycopg2.extensions.cursor, stop_requested: Event):
    # Heartbeating is not essential to the main activity of task processing, so not intercepting exceptions here is OK:
    # in case of any the heartbeating would just be abandoned, the exception --- implicitly logged, the main thread --- left unaffected.
    worker_id: str = env["worker_id"]
    database: Database = env["db"]
    settings: Settings = env["settings"]
    rest_period_in_seconds = float(settings.tasking.heartbeat_rest_period_in_seconds)
    while not stop_requested.wait(rest_period_in_seconds):
        database.refresh_task(task_id=task_id, worker_id=worker_id, cur=cursor)


@contextmanager
def heartbeat(env: Env, task_id: int):
    database: Database = env["db"]
    with closing(database.make_connection()) as connection:
        connection.autocommit = True
        with connection.cursor() as cursor:
            stop_requested = Event()
            thread = Thread(
                name="heartbeat",
                target=_run,
                kwargs=dict(env=env, task_id=task_id, cursor=cursor, stop_requested=stop_requested),
            )
            thread.start()
            try:
                yield
            finally:
                stop_requested.set()
                thread.join()
