from functools import wraps
from pathlib import PurePosixPath
from urllib.parse import urlencode
import urllib3
import requests, requests.adapters
from tractor.secrets import Secrets
from tractor_disk.settings import settings
from tractor_disk.disk_error import DiskTemporaryError, DiskPermanentError, catch_errors
from tractor.util.retrying import need_retry, TemporaryError

DEFAULT_PAGE_SIZE = 100
PATH_PREFIX = "disk:"


DISK_NAME = "YandexDisk"


def catch_disk_errors(method):
    return catch_errors(method, _dispatch_disk_error)


def _dispatch_disk_error(exc: Exception):
    try:
        raise exc
    except (DiskTemporaryError, DiskPermanentError) as disk_err:
        if not disk_err.source:
            disk_err.source = DISK_NAME
        raise disk_err
    except requests.HTTPError as http_err:
        msg = str(http_err)
        if need_retry(http_err.response.status):
            raise DiskTemporaryError(msg, DISK_NAME)
        else:
            raise DiskPermanentError(msg, DISK_NAME)
    except (TemporaryError, TimeoutError, ConnectionError, OSError) as tmp_err:
        raise DiskTemporaryError(str(tmp_err), DISK_NAME)


def decorate_path(method):
    @wraps(method)
    def impl(self, path, *args, **kwargs):
        if not path.startswith(PATH_PREFIX):
            path = PurePosixPath(PATH_PREFIX).joinpath(path)
        return method(self, path, *args, **kwargs)

    return impl


class YandexDisk:
    def __init__(self, uid):
        self._uid = uid
        self._resources_cache = {}
        self._settings = settings().yandex_disk
        self._session = requests.Session()
        retrying_strategy = urllib3.Retry(
            total=self._settings.retrying.count,
            status_forcelist=[429, 500, 502, 503, 504],
            backoff_factor=self._settings.retrying.base_delay_in_seconds,
        )
        adapter = requests.adapters.HTTPAdapter(max_retries=retrying_strategy)
        self._session.mount("http://", adapter)
        self._session.mount("https://", adapter)

    @decorate_path
    def folder_exists(self, path):
        return self._resource_exists(path)

    @decorate_path
    def file_exists(self, path):
        return self._resource_exists(path)

    @decorate_path
    def create_folder(self, path):
        resp = self._api_put("v1/disk/resources", path=path)
        if path in self._resources_cache:
            del self._resources_cache[path]
        if (
            resp.status_code == 409
            and resp.json()["error"] == "DiskPathPointsToExistentDirectoryError"
        ):
            return
        self.raise_for_status(resp)

    @decorate_path
    def store_file(self, path, content):
        upload_url = self._get_url_for_upload(path)
        self._upload(upload_url, path, content)

    def quote(self):
        resp = self._api_get("v1/disk", fields="used_space,total_space")
        self.raise_for_status(resp)
        return resp.json()

    def files(self):
        res = []
        offset = 0
        while True:
            resp = self._api_get(
                "v1/disk/resources/files",
                fields="items.path,items.size",
                limit=DEFAULT_PAGE_SIZE,
                offset=offset,
            )
            self.raise_for_status(resp)
            files = resp.json().get("items")
            if not files:
                break
            res += files
            offset += len(files)
        return res

    def _get_resource(self, path):
        if path in self._resources_cache:
            return self._resources_cache[path]
        resp = self._api_get("v1/disk/resources", path=path)
        if resp.status_code == 404:
            self._resources_cache[path] = None
            return None
        self.raise_for_status(resp)
        self._resources_cache[path] = resp.json()
        return resp.json()

    def _resource_exists(self, path):
        resource = self._get_resource(path)
        return resource is not None

    def _get_url_for_upload(self, path):
        resp = self._api_get("v1/disk/resources/upload", path=path)
        self.raise_for_status(resp)
        resp_json = resp.json()
        if resp_json["method"] != "PUT":
            raise RuntimeError("Method for upload should be PUT")
        if resp_json["templated"]:
            raise RuntimeError("Url for upload should not be templated")
        return resp_json["href"]

    @catch_disk_errors
    def _upload(self, upload_url, path, content):
        resp = self._session.put(
            upload_url, data=content, verify=self._settings.verify_ssl_on_upload
        )
        self.raise_for_status(resp)
        if path in self._resources_cache:
            del self._resources_cache[path]

    @catch_disk_errors
    def _api_get(self, resource, **kwargs):
        url = "{}/{}?{}".format(self._settings.host, resource, urlencode(kwargs))
        headers = {
            "User-Agent": self._settings.user_agent,
            "X-Ya-Service-Ticket": Secrets().yandex_disk_secret(),
            "X-Uid": self._uid,
        }
        resp = self._session.get(url, headers=headers)
        return resp

    @catch_disk_errors
    def _api_put(self, resource, **kwargs):
        url = "{}/{}?{}".format(self._settings.host, resource, urlencode(kwargs))
        headers = {
            "User-Agent": self._settings.user_agent,
            "X-Ya-Service-Ticket": Secrets().yandex_disk_secret(),
            "X-Uid": self._uid,
        }
        resp = self._session.put(url, headers=headers)
        return resp

    @catch_disk_errors
    def raise_for_status(self, resp):
        resp.raise_for_status()


class YandexDiskWithFakeRoot:
    def __init__(self, root, uid):
        self.yandex_disk = YandexDisk(uid)
        self.root = root

    def folder_exists(self, path):
        path = "{}/{}".format(self.root, path)
        return self.yandex_disk.folder_exists(path)

    def file_exists(self, path):
        path = "{}/{}".format(self.root, path)
        return self.yandex_disk.file_exists(path)

    def create_folder(self, path):
        if not self.yandex_disk.folder_exists(self.root):
            self.yandex_disk.create_folder(self.root)
        path = "{}/{}".format(self.root, path)
        self.yandex_disk.create_folder(path)

    def store_file(self, path, content):
        if not self.yandex_disk.folder_exists(self.root):
            self.yandex_disk.create_folder(self.root)
        path = "{}/{}".format(self.root, path)
        self.yandex_disk.store_file(path, content)


def create_yandex_disk(uid):
    return YandexDisk(uid)


def create_yandex_disk_with_fake_root(root, uid):
    return YandexDiskWithFakeRoot(root, uid)
