from contextlib import closing
from typing import Any, Dict
from tractor.error import MAIL_SYNC_ERROR
from tractor.yandex_services.collectors import CollectorInfo, Collectors
from tractor.mail.db import Database
from tractor.mail.models import (
    MessagesCollectionStats,
    UserMigration,
    UserMigrationStatus,
)


Env = Dict[str, Any]


def process_migration(migration: UserMigration, env: Env):
    db: Database = env["db"]

    uid, suid, popid = _get_user_and_collector_ids(migration, env)
    collector_info = _get_collector_info(uid, suid, popid, env)

    if _collector_in_perm_error(collector_info):
        _move_to_error_state(migration, MAIL_SYNC_ERROR, env)
        return

    if _in_initial_sync(migration) and _all_messages_collected(collector_info):
        _move_to_sync_newest_status(migration, env)

    folders_stats = _get_collector_folders_stats(uid, suid, popid, env)
    _update_stats(migration, folders_stats, env)


def _get_user_and_collector_ids(migration: UserMigration, env: Env):
    db: Database = env["db"]
    with closing(db.make_connection()) as conn, conn, conn.cursor() as cur:
        prepare_task = db.get_task_by_task_id(migration.prepare_task_id, cur)

    for field in ["uid", "suid", "popid"]:
        if field not in prepare_task.worker_output:
            raise RuntimeError("no {} in prepare task output".format(field))

    uid = prepare_task.worker_output["uid"]
    suid = prepare_task.worker_output["suid"]
    popid = prepare_task.worker_output["popid"]

    return uid, suid, popid


def _get_collector_info(uid, suid, popid, env: Env) -> CollectorInfo:
    collectors: Collectors = env["collectors"]
    for collector_info in collectors.list(uid, suid):
        if collector_info.popid == popid:
            return collector_info
    raise RuntimeError("popid {} not found in /list response".format(popid))


def _get_collector_folders_stats(uid, suid, popid, env) -> Dict[str, MessagesCollectionStats]:
    collectors: Collectors = env["collectors"]
    stats = collectors.status(uid, suid, popid)
    return stats


def _update_stats(
    migration: UserMigration, folders_stats: Dict[str, MessagesCollectionStats], env: Env
):
    stats = {"folders": {}}
    for folder_name, folder_stats in folders_stats.items():
        stats["folders"][folder_name] = {
            "src_mailbox_messages_count": folder_stats.source_count,
            "collected_messages_count": folder_stats.collected_count,
            "failed_to_collect_messages_count": folder_stats.failed_count,
        }

    db: Database = env["db"]
    with closing(db.make_connection()) as conn, conn, conn.cursor() as cur:
        db.update_stats(migration.org_id, migration.login, stats, cur)


def _collector_in_perm_error(collector_info: CollectorInfo) -> bool:
    return collector_info.is_on == 3


def _move_to_error_state(migration: UserMigration, error_reason: str, env: Env):
    db: Database = env["db"]
    with closing(db.make_connection()) as conn, conn, conn.cursor() as cur:
        db.mark_migration_finished(
            migration.org_id,
            migration.login,
            UserMigrationStatus.ERROR,
            error_reason,
            cur,
        )


def _move_to_sync_newest_status(migration: UserMigration, env: Env):
    db: Database = env["db"]
    with closing(db.make_connection()) as conn, conn, conn.cursor() as cur:
        db.move_migration_to_new_status(
            migration.org_id,
            migration.login,
            migration.status,
            UserMigrationStatus.SYNC_NEWEST,
            cur,
        )


def _in_initial_sync(migration: UserMigration) -> bool:
    return migration.status == UserMigrationStatus.INITIAL_SYNC


def _all_messages_collected(collector_info: CollectorInfo) -> bool:
    return collector_info.error_status == "ok" and collector_info.last_msg_count == 0
