from contextlib import closing
from typing import Any, Dict
from tractor.logger import DeployLogger
from tractor.mail.db import Database
from tractor.mail.models import UserMigration, UserMigrationStatus
from tractor.models import TaskWorkerStatus
from tractor.error import (
    MIGRATION_WAS_STOPPED_BEFORE_IT_WAS_FINISHED,
    PREPARING_ERROR,
    STOPPING_ERROR,
)
from psycopg2.extensions import cursor

Env = Dict[str, Any]


def process_migration_in_preparing_status(env: Env, migration: UserMigration, cur: cursor):
    db: Database = env["db"]
    prepare_task = db.get_task_by_task_id(migration.prepare_task_id, cur)
    if prepare_task.worker_status == TaskWorkerStatus.SUCCESS:
        db.move_migration_to_new_status(
            migration.org_id,
            migration.login,
            migration.status,
            UserMigrationStatus.INITIAL_SYNC,
            cur,
        )
    else:
        db.move_migration_to_new_status(
            migration.org_id,
            migration.login,
            migration.status,
            UserMigrationStatus.ERROR,
            cur,
            error_reason=PREPARING_ERROR,
        )


def process_migration_in_stopping_status(env: Env, migration: UserMigration, cur: cursor):
    db: Database = env["db"]
    stop_task = db.get_task_by_task_id(migration.stop_task_id, cur)
    if stop_task.worker_status == TaskWorkerStatus.ERROR:
        db.move_migration_to_new_status(
            migration.org_id,
            migration.login,
            migration.status,
            UserMigrationStatus.ERROR,
            cur,
            error_reason=STOPPING_ERROR,
        )
    else:
        migration_prev_status = stop_task.input["previous_migration_status"]
        if migration_prev_status != UserMigrationStatus.SYNC_NEWEST.value:
            db.move_migration_to_new_status(
                migration.org_id,
                migration.login,
                migration.status,
                UserMigrationStatus.ERROR,
                cur,
                error_reason=MIGRATION_WAS_STOPPED_BEFORE_IT_WAS_FINISHED,
            )
        else:
            db.move_migration_to_new_status(
                migration.org_id,
                migration.login,
                migration.status,
                UserMigrationStatus.STOPPED,
                cur,
            )
