from base64 import b64decode
from contextlib import closing
import json
from typing import Any, Dict, List, Optional, Tuple
from tractor.crypto.fernet import Fernet
from tractor.logger import DeployLogger
from tractor.mail.db import Database
from tractor.mail.models import ExternalProviderImapServer, MailServerConnectionInfo
from tractor.models import ExternalProvider, Task
from tractor.yandex_services.collectors import CollectorInfo, Collectors
from tractor.yandex_services.directory import Directory
from tractor.yandex_services.blackbox import Blackbox
from tractor.mail.models import MailServerInfo, UserInfo
from tractor.util.dataclasses import construct_from_dict


Env = Dict[str, Any]


def trycatch(method):
    def wrap(task: Task, env: Env):
        try:
            method(task, env)
        except Exception as ex:
            logger: DeployLogger = env["logger"]
            try:
                db: Database = env["db"]
                logger.exception(message="prepare task failed", exception=ex)
                error_msg = _error_message_from_exception(ex)
                error_msg = json.dumps({"error": error_msg})
                with closing(db.make_connection()) as conn, conn, conn.cursor() as cur:
                    db.fail_task(output=error_msg, task_id=task.task_id, cur=cur)
            except Exception as another_e:
                logger.exception(message="cannot fail prepare task", exception=another_e)

    return wrap


def _error_message_from_exception(e):
    return str(e)


@trycatch
def run_task(task: Task, env: Env):
    if task.canceled:
        _set_error_for_cancelled_task(task, env)
        return

    task_input = task.input
    admin_uid, user_info, mail_server_info = _unpack_task_input(env, task_input)
    user_info: UserInfo
    mail_server_info: MailServerInfo
    user_info.uid = _create_user_if_needed(env, admin_uid, user_info)
    suid = _get_suid(env, user_info.uid)

    _delete_old_collectors_if_needed(env, user_info.uid, suid, user_info.login)
    popid = _create_new_collector(
        env,
        user_info.uid,
        suid,
        user_info.login,
        task.domain,
        user_info.external_password,
        mail_server_info,
    )
    _finish_task(env, task.task_id, user_info.uid, suid, popid)


def _unpack_task_input(env: Env, task_input: dict) -> Tuple[str, Dict[str, str], Dict[str, Any]]:
    fernet: Fernet = env["fernet"]
    admin_uid = task_input["admin_uid"]
    user_info = task_input["user"]

    encrypted_external_password = user_info.get("encrypted_external_password")
    if encrypted_external_password is not None:
        user_info["external_password"] = _decrypt_password(fernet, encrypted_external_password)
        del user_info["encrypted_external_password"]

    encrypted_yandex_password = user_info.get("encrypted_yandex_password")
    if encrypted_yandex_password is not None:
        user_info["yandex_password"] = _decrypt_password(fernet, encrypted_yandex_password)
        del user_info["encrypted_yandex_password"]
    user_info = construct_from_dict(UserInfo, task_input["user"])
    mail_server_info_dct = task_input["mail_server_info"]
    mail_server_info = MailServerInfo(mail_server_info_dct["provider"])
    if mail_server_info.provider == "custom":
        mail_server_info.conn_info = MailServerConnectionInfo(
            mail_server_info_dct["host"],
            mail_server_info_dct["port"],
            mail_server_info_dct["ssl"],
        )
    return admin_uid, user_info, mail_server_info


def _decrypt_password(fernet: Fernet, encrepted_password: str) -> str:
    return fernet.decrypt_text(b64decode(encrepted_password))


def _create_user_if_needed(env: Env, admin_uid, user_info: UserInfo) -> str:
    logger: DeployLogger = env["logger"]
    directory: Directory = env["directory"]
    uid = user_info.uid
    if uid is None:
        uid: str = directory.create_user(admin_uid, user_info)
        logger.info(message=f"user with login={user_info.login} was created with uid={uid}")
    return uid


def _get_suid(env: Env, uid: str):
    blackbox: Blackbox = env["blackbox"]
    suid: str = blackbox.get_suid(uid)
    return suid


def _delete_old_collectors_if_needed(env: Env, uid: str, suid: str, login: str):
    collectors: Collectors = env["collectors"]
    logger: DeployLogger = env["logger"]
    existing_collectors: List[CollectorInfo] = collectors.list(uid, suid)
    for collector in existing_collectors:
        collectors.delete(uid, suid, collector.popid)
        logger.info(
            message=f"collector with popid={collector.popid} for user with login={login} was deleted"
        )


def _create_new_collector(
    env: Env,
    uid: str,
    suid: str,
    login: str,
    domain: str,
    password: str,
    mail_server_info: MailServerInfo,
) -> str:
    collectors: Collectors = env["collectors"]
    logger: DeployLogger = env["logger"]
    email = f"{login}@{domain}"
    password, conn_info = _get_info_for_collector_creation(password, mail_server_info)
    popid: str = collectors.create(email, password, uid, suid, conn_info)
    logger.info(message=f"collector for user with login={login} was created with popid={popid}")
    return popid


def _get_info_for_collector_creation(
    password: Optional[str],
    mail_server_info: MailServerInfo,
) -> Tuple[str, MailServerConnectionInfo]:
    if mail_server_info.provider == ExternalProvider.GOOGLE.value:
        return (
            "gmail-oauth2",
            MailServerConnectionInfo(
                host=ExternalProviderImapServer.GOOGLE.value,
                port=993,
                ssl=True,
            ),
        )
    if mail_server_info.provider == ExternalProvider.MICROSOFT.value:
        return (
            "outlook-oauth2",
            MailServerConnectionInfo(
                host=ExternalProviderImapServer.MICROSOFT.value,
                port=993,
                ssl=True,
            ),
        )
    if password is None:
        raise RuntimeError("external password for custom provider is required")
    if mail_server_info.conn_info is None:
        raise RuntimeError("connection info for custom provider is required")
    return password, mail_server_info.conn_info


def _finish_task(env: Env, task_id: int, uid: str, suid: str, popid: str):
    db: Database = env["db"]
    task_output = json.dumps({"uid": uid, "suid": suid, "popid": popid})
    with db.make_connection() as conn:
        with conn.cursor() as cur:
            db.finish_task(task_output, task_id, cur)


def _set_error_for_cancelled_task(task: Task, env: Env) -> bool:
    db: Database = env["db"]
    logger: DeployLogger = env["logger"]
    try:
        with closing(db.make_connection()) as conn, conn, conn.cursor() as cur:
            db.set_error_for_cancelled_task(task.task_id, cur)
    except Exception as ex:
        logger.exception(message="cannot set error for cancelled prepare task", exception=ex)
