from contextlib import closing
import json
from typing import Any, Dict
from tractor.logger import DeployLogger
from tractor.mail.db import Database
from tractor.models import Task, TaskWorkerStatus
from tractor.yandex_services.collectors import Collectors


Env = Dict[str, Any]


def trycatch(method):
    def wrap(task: Task, env: Env):
        try:
            method(task, env)
        except Exception as ex:
            logger: DeployLogger = env["logger"]
            try:
                db: Database = env["db"]
                logger.exception(message="stop task failed", exception=ex)
                error_msg = _error_message_from_exception(ex)
                with closing(db.make_connection()) as conn, conn, conn.cursor() as cur:
                    db.fail_task(
                        output=json.dumps({"error": error_msg}),
                        task_id=task.task_id,
                        cur=cur,
                    )
            except Exception as another_e:
                logger.exception(message="cannot fail stop task", exception=another_e)

    return wrap


def _error_message_from_exception(e):
    return str(e)


@trycatch
def run_task(task: Task, env: Env):
    prepare_task: Task = _get_task_by_id(task.input["prepare_task_id"], env)
    if prepare_task.worker_status == TaskWorkerStatus.PENDING:
        return
    if prepare_task.worker_status == TaskWorkerStatus.SUCCESS:
        _delete_collector_created_in_prepare_task(prepare_task.worker_output, env)
    _finish_task(env, task.task_id)


def _get_task_by_id(prepare_task_id, env: Env) -> Task:
    db: Database = env["db"]
    with closing(db.make_connection()) as conn, conn, conn.cursor() as cur:
        prepare_task = db.get_task_by_task_id(prepare_task_id, cur)
        return prepare_task


def _delete_collector_created_in_prepare_task(prepare_task_output: Dict[str, str], env: Env):
    collectors: Collectors = env["collectors"]
    logger: DeployLogger = env["logger"]
    popid = prepare_task_output["popid"]
    uid = prepare_task_output["uid"]
    suid = prepare_task_output["suid"]
    collectors.delete(uid, suid, popid)
    logger.info(
        message=f"deleted collector with popid={popid} which was created for user with uid={uid}"
    )


def _finish_task(env: Env, task_id: int):
    db: Database = env["db"]
    task_output = json.dumps({})
    with closing(db.make_connection()) as conn, conn, conn.cursor() as cur:
        db.finish_task(task_output, task_id, cur)
