#!/usr/bin/env python3

from copy import copy
from datetime import timedelta
import time
import traceback, signal
from typing import Optional
from tractor.logger import DeployLogger, get_logger, setup_logger
from tractor.yandex_services.directory import Directory
from tractor.yandex_services.collectors import Collectors
from tractor.mail.db import Database
from tractor.mail.models import UserMigration
from tractor_mail.settings import Settings, settings
from tractor_mail.impl.collect_status_updater import Env, process_migration


def main():
    global_env: Env = _make_env()
    settings: Settings = global_env["settings"]
    logger: DeployLogger = global_env["logger"]
    while True:
        migration = _try_acquire_migration(global_env)
        if migration:
            logger.info(
                message="acquired migration for status update",
                org_id=migration.org_id,
                domain=migration.domain,
                login=migration.login,
            )
            migration_env: Env = _make_migration_env(migration, global_env)
            _process_migration(migration, migration_env)
        else:
            logger.info(message="no acquired migration for status update")
            time.sleep(settings.collect_status_update.sleep_period_in_seconds)


def _make_env() -> Env:
    res = {}
    res["settings"] = settings()
    res["db"] = Database(settings().tractor_disk_db)
    res["logger"] = get_logger()
    return res


def _try_acquire_migration(env: Env) -> Optional[UserMigration]:
    logger = env["logger"]
    try:
        settings: Settings = env["settings"]
        db: Database = env["db"]
        with db.make_connection() as conn:
            with conn.cursor() as cur:
                return db.acquire_migration_for_stats_update(
                    update_interval=timedelta(
                        seconds=settings.collect_status_update.migration_update_period_in_seconds
                    ),
                    cur=cur,
                )
    except Exception as e:
        logger.exception("failed to acquire migration for status update", exception=e)
        return None


def _make_migration_env(migration: UserMigration, global_env: Env) -> Env:
    migration_env = copy(global_env)
    migration_env.update(
        {
            "directory": Directory(migration.org_id, settings().directory),
            "collectors": Collectors(settings().collectors),
            "logger": get_logger(
                org_id=migration.org_id, domain=migration.domain, login=migration.login
            ),
        }
    )
    return migration_env


def _process_migration(migration, env: Env):
    logger = env["logger"]
    try:
        process_migration(migration, env)
    except Exception as e:
        logger.exception(
            "failed to process migration status update",
            exception=e,
        )


def _debug(sig, frame):  # TODO
    message = "Signal received. Current traceback:\n"
    message += "".join(traceback.format_stack(frame))
    get_logger().info(message)


def _listen_debug():
    signal.signal(signal.SIGUSR1, _debug)


if __name__ == "__main__":
    _listen_debug()
    setup_logger(settings().logging)
    main()
