from contextlib import closing
import time
from tractor.logger import DeployLogger, get_logger, setup_logger
from tractor.mail.db import Database
from tractor_mail.impl.coordinator import (
    Env,
    process_migration_in_preparing_status,
    process_migration_in_stopping_status,
)
from tractor_mail.settings import Settings, settings


def main():
    env = _make_env()
    while True:
        logger: DeployLogger = env["logger"]
        if _try_acquire_and_process_preparing_user_migration(env):
            continue
        if _try_acquire_and_process_stopping_user_migration(env):
            continue
        logger.info(message="no migrations acquired")
        time.sleep(settings().coordinator.sleep_period_in_seconds)


def trycatch(method):
    def wrap(env: Env):
        try:
            return method(env)
        except Exception as ex:
            logger: DeployLogger = env["logger"]
            logger.exception(
                message=f"error in coordinator ocurred",
                exception=str(ex),
            )
            return False

    return wrap


@trycatch
def _try_acquire_and_process_preparing_user_migration(env: Env):
    db: Database = env["db"]
    logger: DeployLogger = env["logger"]
    with closing(db.make_connection()) as conn, conn, conn.cursor() as cur:
        preparing_migration = db.acquire_migration_in_preparing_status(cur)
        if preparing_migration is None:
            return False
        logger.info(
            message="acquired migration in preparing status",
            org_id=preparing_migration.org_id,
            login=preparing_migration.login,
        )
        process_migration_in_preparing_status(env, preparing_migration, cur)
        return True


@trycatch
def _try_acquire_and_process_stopping_user_migration(env: Env):
    db: Database = env["db"]
    logger: DeployLogger = env["logger"]
    with closing(db.make_connection()) as conn, conn, conn.cursor() as cur:
        stopping_migration = db.acquire_migration_in_stopping_status(cur)
        if stopping_migration is None:
            return False
        logger.info(
            message="acquired migration in stopping status",
            org_id=stopping_migration.org_id,
            login=stopping_migration.login,
        )
        process_migration_in_stopping_status(env, stopping_migration, cur)
        return True


def _make_env():
    res = {}
    res["db"] = Database(settings().tractor_disk_db)
    res["logger"] = get_logger()
    return res


if __name__ == "__main__":
    setup_logger(Settings().logging)
    main()
