from copy import copy
from datetime import timedelta
from os import getenv, getpid, uname
import time
import traceback, signal
from typing import Any, Dict, Optional

from tractor.logger import DeployLogger, get_logger, setup_logger
from tractor.mail.db import Database
from tractor.mail.models import TaskType
from tractor.models import Task
from tractor.yandex_services.collectors import Collectors
from tractor_mail.settings import Settings, settings
from tractor_mail.impl.stop_worker import run_task


Env = Dict[str, Any]


def main():
    global_env: Env = _make_env()
    settings: Settings = global_env["settings"]
    logger: DeployLogger = global_env["logger"]
    while True:
        task = _try_acquire_task(global_env)
        if task:
            logger.info(message="acquired stop task", task_id=task.task_id)
            task_env: Env = _make_task_env(task, global_env)
            _process_task(task, task_env)
        else:
            logger.info(message="no stop tasks acquired")
            time.sleep(settings.stop_worker_settings.worker_sleep_period_in_seconds)


def _process_task(task, global_env: Env):
    logger: DeployLogger = global_env["logger"]
    try:
        task_env = _make_task_env(task, global_env)
        if task_env:
            run_task(task, task_env)
    except Exception as e:
        logger.exception(
            "failed to process stop task",
            exception=e,
        )


def _try_acquire_task(env: Env) -> Optional[Task]:
    logger: DeployLogger = env["logger"]
    try:
        settings: Settings = env["settings"]
        db: Database = env["db"]
        with db.make_connection() as conn:
            with conn.cursor() as cur:
                return db.acquire_task(
                    type=TaskType.STOP,
                    expiry_timeout=timedelta(seconds=settings.stop_worker_settings.expiry_timeout_in_seconds),
                    worker_id=env["worker_id"],
                    cur=cur,
                )
    except Exception as e:
        logger.exception("failed to acquire prepare task", exception=e)
        return None


def _debug(sig, frame):  # TODO
    message = "Signal received. Current traceback:\n"
    message += "".join(traceback.format_stack(frame))
    get_logger().info(message)


def _listen_debug():
    signal.signal(signal.SIGUSR1, _debug)


def _make_task_env(task: Task, global_env: Env) -> Env:
    task_env = copy(global_env)
    task_env.update(
        {
            "collectors": Collectors(settings().collectors),
            "logger": get_logger(task_id=task.task_id, org_id=task.org_id),
        }
    )
    return task_env


def _make_env() -> Env:
    res = {}
    res["worker_id"] = getenv("DEPLOY_POD_TRANSIENT_FQDN", uname().nodename) + "/" + str(getpid())
    res["settings"] = settings()
    res["db"] = Database(settings().tractor_disk_db)
    res["logger"] = get_logger()
    return res


if __name__ == "__main__":
    _listen_debug()
    setup_logger(settings().logging)
    main()
