from dataclasses import dataclass
from tractor.settings import (
    BlackboxSettings,
    DirectorySettings,
    RetryingSettingsWithDelay,
    Settings as BaseSettings,
    TractorDBSettings,
)
from tractor.util.dataclasses import construct_from_dict


@dataclass
class CollectorsSettings:
    host: str
    timeout: int = 5
    user_agent: str = "tractor"
    retrying: RetryingSettingsWithDelay = RetryingSettingsWithDelay(
        base_delay_in_seconds=0.100, count=3
    )


@dataclass
class LoggingSettings:
    file: str = None


@dataclass
class WorkerSettings:
    worker_sleep_period_in_seconds: int = 60
    expiry_timeout_in_seconds: int = 10 * 60


@dataclass
class CoordinatorSettings:
    sleep_period_in_seconds: int = 60


@dataclass
class CollectStatusUpdateSettings:
    sleep_period_in_seconds: int = 60
    migration_update_period_in_seconds: int = 20 * 60


class Settings(BaseSettings):
    def __init__(self) -> None:
        super().__init__()

        self.tractor_disk_db = construct_from_dict(TractorDBSettings, self.data.get("tractor_db"))
        self.prepare_worker_settings: WorkerSettings = construct_from_dict(
            WorkerSettings, self.data.get("prepare")
        )
        self.stop_worker_settings: WorkerSettings = construct_from_dict(
            WorkerSettings, self.data.get("stop")
        )
        self.directory: DirectorySettings = construct_from_dict(
            DirectorySettings, self.data.get("directory")
        )
        self.blackbox: BlackboxSettings = construct_from_dict(
            BlackboxSettings, self.data.get("blackbox")
        )
        self.collectors = construct_from_dict(CollectorsSettings, self.data.get("collectors"))

        self.logging: LoggingSettings = construct_from_dict(
            LoggingSettings, self.data.get("logging", {})
        )

        self.collect_status_update: CollectStatusUpdateSettings = construct_from_dict(
            CollectStatusUpdateSettings, self.data.get("collect_status_update")
        )

        self.coordinator: CoordinatorSettings = construct_from_dict(
            CoordinatorSettings, self.data.get("coordinator")
        )


def settings() -> Settings:
    if not hasattr(settings, "instance"):
        settings.instance = Settings()

    return settings.instance
