from datetime import datetime
import argparse


class BaseArgument(object):
    def __init__(self, name, help_):
        self.name = name
        self.help_ = help_

    def make_args(self):  # pylint: disable=R0201
        return {}

    def add_to(self, parser):
        parser.add_argument(
            self.name,
            help=self.help_,
            **self.make_args()
        )


class Argument(BaseArgument):
    def __init__(  # pylint: disable=R0913
            self, name, help_,
            metavar=None, type_=str, required=False, **kwargs):
        super(Argument, self).__init__(name, help_)
        self.metavar = metavar or name.lstrip('-').upper()
        self.type_ = type_
        self.required = required
        self.kwargs = kwargs

    def as_required(self):
        self.required = True
        return self

    def make_args(self):
        kwargs = dict(
            metavar=self.metavar,
            type=self.type_,
        )
        if self.name.startswith('-'):
            kwargs['required'] = self.required
        kwargs.update(self.kwargs)
        return kwargs


class Flag(BaseArgument):
    def __init__(self, name, help_):
        super(Flag, self).__init__(name, help_)

    def make_args(self):
        return dict(action='store_true')


class DateArgument(BaseArgument):
    DATE_FORMAT = '%Y-%m-%d'

    def __init__(self, name, help_):
        super(DateArgument, self).__init__(name, help_)

    @classmethod
    def valid_date(cls, s):
        try:
            return datetime.strptime(s, cls.DATE_FORMAT)
        except ValueError:
            raise argparse.ArgumentTypeError(
                "Not a valid date: '{0}'."
                " Expect {1} format. Today is {2}".format(
                    s, cls.DATE_FORMAT, datetime.now().strftime(
                        cls.DATE_FORMAT)))

    def make_args(self):
        return dict(type=self.valid_date)


class Any(object):
    def __init__(self, *arguments):
        self.arguments = arguments

    def add_to(self, parser):
        group = parser.add_mutually_exclusive_group(required=True)
        for arg in self.arguments:
            arg.requred = False
            arg.add_to(group)


class All(object):
    def __init__(self, *arguments):
        self.arguments = arguments

    def add_to(self, parser):
        for arg in self.arguments:
            arg.required = True
            arg.add_to(parser)


class Command(object):
    def __init__(self, name, hlp, *arguments):
        self.name = name
        self.hlp = hlp
        self.arguments = arguments

    def add_to(self, subparsers):
        parser = subparsers.add_parser(self.name, help=self.hlp)
        for arg in self.arguments:
            arg.add_to(parser)


class CommandGroup(object):
    def __init__(self, *arguments):
        self.arguments = arguments

    def add_to(self, parser):
        subparsers = parser.add_subparsers(dest='command_name')
        for arg in self.arguments:
            assert isinstance(arg, Command), \
                'CommandGroup must contain only Command objects'
            arg.add_to(subparsers)


def make_uid_argument(name):
    return Argument(
        name,
        'user ID', type_=int,
        metavar='UID'
    )


Uid = make_uid_argument('--uid')


def build_parser(parser, arguments):
    for arg in arguments:
        arg.add_to(parser)
    return parser
