import keyword
import os.path
import re
from datetime import timedelta

import yaml

from mail.pypg.pypg.arcadia import is_arcadia
from ora2pg.compare import is_seq
from ora2pg.storage import MulcaGate

_FILE_ABS_PATH = os.path.abspath(__file__)


class AllFrom(tuple):
    def why_not_good(self, keys):
        for my_key in self:
            if isinstance(my_key, str):
                if my_key not in keys:
                    return "require %r from %r" % (
                        my_key, self)
            else:
                bad = my_key.why_not_good(keys)
                if bad:
                    return bad

    def __repr__(self):
        return 'AllFrom(%s)' % ' AND '.join([repr(k) for k in self])


class OneFrom(tuple):
    def why_not_good(self, keys):
        found_keys = []
        for my_key in self:
            if isinstance(my_key, str):
                if my_key in keys:
                    found_keys.append(my_key)
            else:
                if not my_key.why_not_good(keys):
                    found_keys.append(my_key)
        if len(found_keys) != 1:
            return 'require one key form %r - found %d' % (
                self, len(found_keys))

    def __repr__(self):
        return 'OneFrom(%s)' % ' OR '.join([repr(k) for k in self])


REQUIRED_CONFIG_ITEMS = (
    'blackbox',
    'mailhost',
    'account_register',
    'sharpei',
    'maildb_dsn_suffix',
    'sharddb_dsn_suffix',
    'huskydb',
    'per_user_logs',
    'mulcagate',
    'reminders',
    'york',
)

CONFIG_EXT = '.yaml'

ARCADIA_CONFIG_BASE = 'resfs/file/mail/transfer/ora2pg/configs/'


class ConfigException(RuntimeError):
    pass


class MalformedConfig(ConfigException):
    pass


class ConfigNotExists(ConfigException):
    pass


def find_base_configs_dir():
    if is_arcadia():
        return ARCADIA_CONFIG_BASE
    'return path to ../configs'
    base_configs_dirs = os.path.abspath(
        os.path.join(
            os.path.dirname(_FILE_ABS_PATH),
            os.path.pardir,
            'configs'))
    if not os.path.exists(base_configs_dirs):
        raise ConfigException(
            "Base_configs_dir %s does not exists" % base_configs_dirs
        )
    return base_configs_dirs


def list_configs_env():
    'return file names without CONFIG_EXT from find_base_configs_dir()'
    if is_arcadia():
        from library.python import resource
        files = list(resource.iterkeys(ARCADIA_CONFIG_BASE, strip_prefix=True))
    else:
        files = os.listdir(find_base_configs_dir())
    files = (f for f in files if f.endswith(CONFIG_EXT))
    return [f[:f.rfind(CONFIG_EXT)] for f in files]


def env_to_config_file(env):
    if is_arcadia():
        base = ARCADIA_CONFIG_BASE
    else:
        base = find_base_configs_dir()
    return os.path.join(base, env + CONFIG_EXT)


def why_bad_config_key(key):
    if not isinstance(key, str):
        return 'can only be as string'
    if keyword.iskeyword(key):
        return 'can not be a python keyword'
    if re.match('[a-z_][a-z0-9_]+$', key, re.I) is None:
        return 'can only contain alphanumeric characters and underscores'


def why_bad_config_value(key, value):
    if key == 'delete_user_shift' and not isinstance(value, timedelta):
        return 'can only be timedelta'


def why_miss_required(config_keys, required_items):
    for key in config_keys:
        bad = why_bad_config_key(key)
        if bad:
            return bad
    for req_key in required_items:
        if isinstance(req_key, str):
            if req_key not in config_keys:
                return 'missed configuration key %r, expect: %r, got: %r' % (
                    req_key, required_items, config_keys)
        else:
            bad = req_key.why_not_good(config_keys)
            if bad:
                return bad


def why_bad_config(config, required_items=REQUIRED_CONFIG_ITEMS):
    try:
        config_keys = config.keys()
    except AttributeError as exc:
        return 'expect dict in config got %r: %s' % (
            type(config), exc)

    why = why_miss_required(config_keys, required_items)
    if why:
        return why

    for k, v in config.items():
        why = why_bad_config_key(k)
        if why:
            return 'bad config key %r: %s' % (k, why)
        why = why_bad_config_value(k, v)
        if why:
            return 'bad config key value %r (%r): %s' % (k, v, why)


def read_config_file(filename):
    if is_arcadia() and filename.startswith('resfs/file'):
        from library.python import resource
        queries_file = resource.find(filename).decode('utf-8')
        if not queries_file:
            raise ConfigNotExists('Config %r do not exists' % filename)
        return queries_file
    else:
        if not os.path.exists(filename):
            raise ConfigNotExists('Config %r do not exists' % filename)
        with open(filename) as fd:
            return fd.read()


def yaml_read_config(path, filename):
    add_custom_yaml_constructors()
    try:
        return yaml.unsafe_load(path)
    except yaml.YAMLError as exc:
        raise MalformedConfig('Broken config %r: %s' % (filename, exc))


def check_config(config, required_items, filename):
    really_bad_config = why_bad_config(config, required_items)
    if really_bad_config:
        raise MalformedConfig(really_bad_config + ' ' + str(filename))


def reformat_mulcagate(config):
    if 'mulcagate' not in config.keys():
        return config
    mulcagate = MulcaGate(
        host=config['mulcagate'],
        port=config.get('mulcagate_port'),
        mg_ca_path=config.get('mulcagate_ca_path')
    )
    config.pop('mulcagate_port', None)
    config.pop('mulcagate_ca_path', None)
    config['mulcagate'] = mulcagate
    return config


def parse_config_file(path, filename, required_items=REQUIRED_CONFIG_ITEMS):
    config = yaml_read_config(path, filename)
    config = reformat_mulcagate(config)
    check_config(config, required_items, filename)
    return config


def load_config_file(filename, required_items=REQUIRED_CONFIG_ITEMS):
    path = read_config_file(filename)
    return parse_config_file(path, filename, required_items)


def load_configs(filenames, required_items=REQUIRED_CONFIG_ITEMS):
    config = {}
    for filename in filenames:
        path = read_config_file(filename)
        config.update(yaml_read_config(path, filename))
    check_config(config, required_items, filenames)
    return config


def timedelta_yaml_constructor(loader, node):
    return timedelta(**loader.construct_mapping(node))


def add_custom_yaml_constructors():
    yaml.add_constructor('!TimeDelta', timedelta_yaml_constructor, Loader=yaml.UnsafeLoader)


add_custom_yaml_constructors()

APP_CONF_PARAM = OneFrom(('env', 'config_path'))


def get_config(config_path, required_items=(APP_CONF_PARAM,)):
    if is_seq(config_path):
        return load_configs(config_path, required_items)
    return load_config_file(config_path, required_items)
