import sys
import os
from functools import wraps

from colorlog import ColoredFormatter

from mail.pypg.pypg.connect import LOG_SQL

import logging
from logging.handlers import WatchedFileHandler

LOG_OSQL = LOG_SQL + 1
logging.addLevelName(LOG_OSQL, 'OSQL')

log = logging.getLogger(__name__)


def logged(func):
    @wraps(func)
    def wrap(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except Exception as exc:
            logging.getLogger().exception(
                '%s throw %s', func.__name__, exc
            )
            raise
    return wrap


def init_main_handler(filepath=None):
    want_colors = filepath is None and sys.stderr.isatty() or os.environ.get('FORCE_COLORS')
    if want_colors:
        formatter = ColoredFormatter(
            "%(asctime)s %(log_color)s%(levelname)-8s %(process)d %(filename)s:"
            "%(lineno)d %(module)s.%(funcName)s: %(message)s",
            log_colors={
                'SQL':      'cyan',
                'OSQL':     'blue',
                'INFO':     'green',
                'WARNING':  'yellow',
                'ERROR':    'red',
                'CRITICAL': 'red'
            }
        )
    else:
        formatter = logging.Formatter(
            "%(asctime)s %(levelname)-8s %(process)d %(filename)s:"
            "%(lineno)d %(module)s.%(funcName)s: %(message)s",
        )

    if filepath is None:
        handler = logging.StreamHandler(sys.stderr)
    else:
        handler = WatchedFileHandler(filepath, mode='a', encoding='utf-8')
    handler.setFormatter(formatter)
    return handler


def init_logging(level, filepath=None):
    logger = logging.getLogger()
    stream_handler = init_main_handler(filepath)
    logger.setLevel(level)
    logger.addHandler(stream_handler)


def current_user_is_root():
    return os.getuid() == 0


def make_dirs_with_perms(base_dir, full_path):
    base_dir_st = os.stat(base_dir)
    if not os.path.exists(full_path):
        os.mkdir(full_path, base_dir_st.st_mode)
    if current_user_is_root():
        os.chown(full_path, base_dir_st.st_uid, base_dir_st.st_gid)


def get_paths_to_uid_dir(uid, base_dir):
    uid_str = '{0:0<6}'.format(uid)
    paths = [
        [uid_str[:3]],
        [uid_str[:3], uid_str[3:]]
    ]
    return [
        os.path.join(base_dir, *p)
        for p in paths
    ]


def tree_by_uid(uid, base_dir):
    full_paths = get_paths_to_uid_dir(uid, base_dir)
    for p in full_paths:
        make_dirs_with_perms(base_dir, p)
    return full_paths[-1]


def init_user_log_handler(uid, base_dir, log_level):
    full_log_path = os.path.join(
        tree_by_uid(uid, base_dir),
        '%d-%s.log' % (
            uid,
            os.path.basename(sys.argv[0])
        )
    )
    handler = logging.FileHandler(
        full_log_path,
        encoding='utf-8',
    )
    handler.setLevel(log_level)
    handler.setFormatter(logging.Formatter(
        "%(asctime)s %(levelname)-8s %(process)d %(filename)s:"
        "%(lineno)d %(module)s.%(funcName)s: %(message)s",
    ))
    return handler
