import logging
import os.path
from argparse import ArgumentParser
from collections import namedtuple

from .arguments import build_parser
from .config_file import list_configs_env, load_config_file, env_to_config_file
from .logs import init_logging, init_user_log_handler

log = logging.getLogger(__name__)


def make_common_parser(description):
    parser = ArgumentParser(
        description=description
    )
    parser.add_argument(
        '-v', '--verbose',
        action='store_true',
        help='print debug'
    )
    config_group = parser.add_mutually_exclusive_group(
        required=True)
    config_group.add_argument(
        '-e', '--environment',
        choices=list_configs_env(),
        help='Environment name',
    )
    config_group.add_argument(
        '--config',
        metavar='FILE',
        help='config file'
    )
    return parser


def join_arguments(config, console):
    res = config.copy()
    for name, value in console.items():
        if name not in res or value is not None:
            res[name] = value
    return res


def dict2args(d):
    Args = namedtuple('Args', d.keys())
    return Args(**d)


def parse_arguments(description, arguments, sys_argv=None):
    parser = make_common_parser(description)
    build_parser(parser, arguments)
    console_args = parser.parse_args(sys_argv)

    config_file_path = console_args.config or env_to_config_file(console_args.environment)

    config = load_config_file(config_file_path)

    j_dict = join_arguments(config, console_args.__dict__)
    return dict2args(j_dict)


class TransferApp(object):

    def __init__(self, args):
        self.args = args
        self._ph = None

    def _get_main_log_level(self):
        level = getattr(self.args, 'log_level', None)
        if level is not None:
            return level
        return logging.DEBUG if self.args.verbose else logging.INFO

    def _get_user_log_level(self):
        level = getattr(self.args, 'user_log_level', None)
        if level is not None:
            return level
        return logging.DEBUG if self.args.verbose else logging.INFO

    def _need_user_log(self):
        need_user_log = getattr(self.args, 'need_user_log', None)
        if need_user_log is not None:
            return need_user_log
        return True

    def init_logging(self):
        init_logging(level=self._get_main_log_level(), filepath=getattr(self.args, 'log_filepath', None))

    def _init_per_user_logging(self, uid):
        if not self._need_user_log():
            return
        if not self.args.per_user_logs:
            log.error('no per_user_logs defined: %r', self.args.per_user_logs)
            return
        root_logger = logging.getLogger()
        self._ph = init_user_log_handler(uid, self.args.per_user_logs, self._get_user_log_level())
        root_logger.addHandler(self._ph)

    def _fini_per_user_logging(self):
        if not self.args.per_user_logs:
            log.error('no per_user_logs defined: %r', self.args.per_user_logs)
            return
        root_logger = logging.getLogger()
        if self._ph is not None:
            root_logger.removeHandler(self._ph)
            self._ph.close()
            self._ph = None

    def set_context(self, user, extra_user=None, extra_tags=None):
        self._fini_per_user_logging()
        self._init_per_user_logging(user.uid)

    def unset_context(self):
        self._fini_per_user_logging()


def make_app(description, *arguments):
    args = parse_arguments(description, arguments)
    tap = TransferApp(args)
    tap.init_logging()
    return tap


def make_app_from_env(environment, **kwargs):
    if os.path.exists(environment) and os.path.isfile(environment):
        return make_app_from_conf(environment, **kwargs)
    return make_app_from_conf(env_to_config_file(environment), **kwargs)


def make_app_from_conf(config_file, **kwargs):
    config = load_config_file(config_file)
    config.update(kwargs)
    if 'verbose' not in config:
        config['verbose'] = False
    tap = TransferApp(dict2args(config))
    tap.init_logging()
    return tap
