from __future__ import print_function

import logging

from collections import namedtuple
from contextlib import closing
from future.moves.urllib.request import urlopen, Request
from future.moves.urllib.parse import urlencode
from typing import Dict, Optional

import lxml.etree as ET

log = logging.getLogger(__name__)


class BlackboxError(Exception):
    pass


class BlackboxCallError(BlackboxError):
    pass


class BlackboxResponseError(BlackboxError):
    pass


class NonExistentUserError(BlackboxError):
    pass


class NotAMailUserError(NonExistentUserError):
    pass


class BBInfo(namedtuple('BBInfo', ('login', 'uid', 'suid', 'default_email', 'attributes'))):
    def __new__(cls, attributes=None, *args, **kwargs):
        return super(BBInfo, cls).__new__(cls, attributes=attributes or [], *args, **kwargs)

    @property
    def is_maillist(self):
        return '13' in self.attributes


def tree2str(tree):
    return ET.tostring(tree.getroot())


class BlackBoxRequest(object):
    TIMEOUT = 5

    def __init__(self, blackbox, tvm_ticket=None):
        # type: (str, Optional[str]) -> None
        self.blackbox = blackbox
        self.url = None
        self.tvm_ticket = tvm_ticket

    def _parse_bb(self, tree):
        # type: (ET.ElementTree) -> BBInfo
        error = tree.find('error')
        if error is not None:
            raise BlackboxResponseError(
                'Error: %r, url: %r response: %s' % (
                    error.text, self.url, tree2str(tree)))
        uid = tree.find('uid').text
        login_elem = tree.find('login')
        if not uid and login_elem is None:
            raise NonExistentUserError(
                'User not exists, url: %r response: %s' % (
                    self.url, tree2str(tree)))

        uid = int(uid)
        login = login_elem.text.replace(u'.', u'-')
        suid = None
        default_email = None

        for field in tree.findall('dbfield'):
            field_type = field.get('id')
            if field_type == 'subscription.suid.2':
                try:
                    suid = int(field.text)
                except (TypeError, ValueError):
                    suid = None
        if suid is None:
            raise NotAMailUserError(
                'Not a mail user, suid: %r, url: %r response: %s' % (
                    suid, self.url, tree2str(tree)))
        for address in tree.find('address-list').iterfind('address'):
            if address.get('default') == '1':
                default_email = address.text
                break

        attributes = []
        attr_node = tree.find('attributes')
        if attr_node is not None:
            attributes = [attr.get('type') for attr in attr_node.iterfind('attribute')]

        return BBInfo(uid=uid, suid=suid, login=login,
                      default_email=default_email, attributes=attributes)

    def _mk_url(self, by_args):
        # type: (Dict) -> str
        args = dict(
            method='userinfo',
            userip='127.0.0.1',
            sid=2,
            dbfields='subscription.suid.2',
            attributes='13',
            emails='getdefault',
        )
        args.update(by_args)
        self.url = self.blackbox + '?' + urlencode(args)
        log.debug('blackbox url: %r', self.url)
        return self.url

    def _headers(self):
        # type: () -> Dict[str, str]
        if self.tvm_ticket is not None:
            headers = {'X-Ya-Service-Ticket': self.tvm_ticket}
        else:
            headers = {}
        return headers

    def _black_request(self, **by_args):
        # type: (Dict[str, str]) -> BBInfo
        try:
            request = Request(self._mk_url(by_args), headers=self._headers())
            with closing(
                urlopen(request, timeout=self.TIMEOUT)
            ) as fd:
                try:
                    tree = ET.parse(fd)
                except SyntaxError as exc:
                    raise BlackboxResponseError(
                        'Malformed response: %s' % exc
                    )
                return self._parse_bb(tree)
        except IOError as exc:
            raise BlackboxCallError(
                'Error: %r, url: %r' % (exc, self.url))
        except RuntimeError as exc:
            log.error('Blackbox request failed, args: %r', by_args)
            raise exc

    def by_login(self, login):
        # type: (str) -> BBInfo
        assert login, 'bad login %r' % login
        return self._black_request(sid='mail', login=login)

    def by_suid(self, suid):
        # type: (int) -> BBInfo
        assert suid, 'bad suid %r' % suid
        return self._black_request(suid=str(suid))

    def by_uid(self, uid):
        # type: (int) -> BBInfo
        assert uid, 'bad uid %r' % uid
        return self._black_request(uid=str(uid))


def get_all_by_login(blackbox, login, tvm_ticket=None):
    # type: (str, str, Optional[str]) -> BBInfo
    return BlackBoxRequest(blackbox, tvm_ticket).by_login(login)


def get_all_by_uid(blackbox, uid, tvm_ticket=None):
    # type: (str, int, Optional[str], Optional[str]) -> BBInfo
    return BlackBoxRequest(blackbox, tvm_ticket).by_uid(uid)
