import logging

from ora2pg.blackbox import get_all_by_uid, NonExistentUserError
from ora2pg.sharpei import get_connstring_by_id
from mail.pypg.pypg.common import qexec, get_connection
from mail.pypg.pypg.query_conf import load_from_package

log = logging.getLogger(__name__)
Q = load_from_package(__package__, __file__)


def get_all_shard_ids(conn):
    cur = qexec(conn, Q.get_all_shard_ids)
    return [s[0] for s in cur]


def get_shard_users_locked(conn, shard_id):
    cur = qexec(conn, Q.get_users_from_sharddb_locked, shard_id=shard_id)
    return [u[0] for u in cur]


def get_absent_in_shard(conn, uids):
    cur = qexec(conn, Q.get_absent_in_shard, uids=list(uids))
    return [u[0] for u in cur]


def worker(blackbox, in_queue, out_queue):
    while not in_queue.empty():
        uid = in_queue.get()
        try:
            get_all_by_uid(blackbox, uid)
        except NonExistentUserError:
            log.warning(
                'User uid={0} from not found in blackbox!'.format(uid)
            )
            out_queue.put(uid)
        except BaseException as e:
            log.exception(e)


def get_absent_in_blackbox(blackbox, uids, shard_id, worker_count):
    from threading import Thread
    import queue

    in_queue = queue.Queue()
    out_queue = queue.Queue()

    for uid in uids:
        in_queue.put(uid)

    workers = []
    for _ in range(worker_count):
        w = Thread(target=worker, args=[blackbox, in_queue, out_queue])
        w.start()
        workers.append(w)
    for w in workers:
        w.join()

    absent_in_blackbox = []

    while not out_queue.empty():
        absent_in_blackbox.append(out_queue.get())

    return absent_in_blackbox


def get_absent_uids(all_uids, args, shard_id, workers):
    log.info('Processing shard: %r', shard_id)
    maildb_dsn = get_connstring_by_id(
        args.sharpei,
        shard_id,
        args.maildb_dsn_suffix
    )

    absent_in_blackbox = get_absent_in_blackbox(args.blackbox, all_uids, shard_id, workers)
    log.debug('Absent in blackbox (mdb!=pg): {0}'.format(absent_in_blackbox))

    with get_connection(maildb_dsn) as maildb_conn:
        absent_in_shard = get_absent_in_shard(maildb_conn, all_uids)
        log.debug('Absent in shard_id={0}: {1}'.format(absent_in_shard, shard_id))
        # Users are not in shard, but bb + sharpei says that they are
        broken_users = set(absent_in_shard) - set(absent_in_blackbox)
        if broken_users:
            log.error('''Shard #{0} says theese users aren't there, but bb+sharpei says they are: {1}'''.format(
                shard_id, broken_users
            ))

    log.info(
        'Finish shard: %r, %d users are absent!',
        shard_id,
        len(absent_in_blackbox)
    )
    return absent_in_blackbox


def remove_absent_uids(conn, uids, shard_id):
    log.debug(
        'Removing users of shard_id={0} from shards.users: {1}'.format(
            shard_id,
            uids
        )
    )
    qexec(conn, Q.remove_absent_uids, shard_id=shard_id, uids=uids)


def clean(args, workers=8, sharddb_dsn=None, shard_ids=None):
    if sharddb_dsn is None:
        sharddb_dsn = args.sharddb
    with get_connection(sharddb_dsn) as sharddb_conn:
        shard_ids = shard_ids or get_all_shard_ids(sharddb_conn)
        for shard_id in shard_ids:
            with sharddb_conn as sharddb_transaction_conn:
                all_uids = get_shard_users_locked(sharddb_transaction_conn, shard_id)
                absent_uids = get_absent_uids(all_uids, args, shard_id, workers)
                remove_absent_uids(sharddb_transaction_conn, absent_uids, shard_id)
