import logging
from contextlib import contextmanager

from mail.pypg.pypg.common import autocommit_connection as pg_autocommit_connection
from mail.pypg.pypg.common import transaction as pg_transaction

from ora2pg.disable_filters import disable_filters_without_type
from ora2pg.pg_put import put_user, write_transfer_info
from ora2pg.sharpei import (
    get_shard_info,
    get_connstring_by_id,
)
from ora2pg.transfer_data import (
    get_user_in_endpoint,
    get_transfer_info,
    DbEndpoint,
)
from pymdb.types import FilterActionOperType
from .source_user_fetcher import PGSourceFetcher
from .stids_copier import STIDsCopier
from .stids_copier_cache import STIDsCopierCache
from .paranoia import NotSupportedUserError

log = logging.getLogger(__name__)

SAFE_FILTER_ACTIONS_TYPES = [
    FilterActionOperType.move,
    FilterActionOperType.movel,
    FilterActionOperType.status,
]


def override_state(user_data):
    if user_data.user.state in ('active', 'inactive', 'notified', 'frozen'):
        user_data.user.state = 'special'
        user_data.user.notifies_count = -1


def override_stids(stids_copier, user_data):
    def rewrite_stids(orig_generator):
        for m in orig_generator:
            stid_with_mime = stids_copier.copy(m.coords.st_id)
            m.coords.st_id = stid_with_mime.st_id
            if stid_with_mime.mime:
                m.mime = stid_with_mime.mime
            yield m

    user_data.mails = rewrite_stids(user_data.mails)
    user_data.deleted_mails = rewrite_stids(user_data.deleted_mails)


def get_pg_dsn_by_shard_id(shard_id, config):
    return get_connstring_by_id(
        sharpei=config.sharpei,
        shard_id=shard_id,
        dsn_suffix=config.maildb_dsn_suffix)


def get_source_fetcher(source_user, config, min_received_date, max_received_date):
    shard_id = get_shard_info(
        uid=source_user.uid,
        dsn=config.sharddb).shard_id
    return PGSourceFetcher(
        user=source_user,
        shard_id=shard_id,
        pg_dsn=get_pg_dsn_by_shard_id(shard_id, config),
        min_received_date=min_received_date,
        max_received_date=max_received_date,
    )


@contextmanager
def pg_transaction_by_shard_id(shard_id, config):  # pragma: noqa
    with pg_transaction(
        get_pg_dsn_by_shard_id(
            shard_id, config)
    ) as pg_conn:
        yield pg_conn


def write_clone_info(
        dest_pg_conn, source_fetcher,
        dest_user, source_user, config):  # pragma: noqa
    write_transfer_info(
        conn=dest_pg_conn,
        transfer_info=get_transfer_info(
            src=source_fetcher.user_in_endpoint(),
            dst=get_user_in_endpoint(
                db_endpoint=DbEndpoint.make_pg(config.dest_shard_id),
                uid=dest_user.uid
            )
        ),
        uid=dest_user.uid,
    )


def prefetch_stids(huskydb_autocommit_conn, dest_user, source_fetcher, mulcagate):
    log.info('Fill stids cache')
    cache = STIDsCopierCache(huskydb_autocommit_conn, dest_user.uid)
    copier = STIDsCopier(
        mulcagate=mulcagate,
        user=str(dest_user.uid),
        cache=cache,
    )
    with source_fetcher.fetch() as user_data:
        for m in user_data.mails:
            copier.copy(m.coords.st_id)
        for m in user_data.deleted_mails:
            copier.copy(m.coords.st_id)
    return copier, cache


def clone_user_data(source_user, dest_user, config, min_received_date, max_received_date):
    # use config.dest_shard_id for mdb-dsn lookup,
    # cause by uid sharpei can return `old` data
    with pg_transaction_by_shard_id(config.dest_shard_id, config) as dest_pg_conn, \
         pg_autocommit_connection(config.huskydb) as huskydb_conn:

        # create stids copies before start real transfer, cause:
        #  * st_ids coping taskes a lot of time
        #  * don't want to lock dest_user
        #  * there are problems with INSERT INTO change_log on day change
        stids_copier, stids_copier_cache = prefetch_stids(
            huskydb_autocommit_conn=huskydb_conn,
            dest_user=dest_user,
            source_fetcher=get_source_fetcher(
                source_user=source_user,
                config=config,
                min_received_date=min_received_date,
                max_received_date=max_received_date,
            ),
            mulcagate=config.mulcagate,
        )

        try:

            source_fetcher = get_source_fetcher(
                source_user=source_user,
                config=config,
                min_received_date=min_received_date,
                max_received_date=max_received_date,
            )

            with source_fetcher.fetch() as dest_data:
                override_state(dest_data)
                if dest_data.state == 'archived':
                    raise NotSupportedUserError('Source user already archived')

                override_stids(
                    stids_copier=stids_copier,
                    user_data=dest_data,
                )

                disable_filters_without_type(dest_data, SAFE_FILTER_ACTIONS_TYPES)

                put_user(
                    user=dest_data,
                    uid=dest_user.uid,
                    conn=dest_pg_conn,
                    fill_changelog=True,
                )

                write_clone_info(
                    dest_pg_conn=dest_pg_conn,
                    source_fetcher=source_fetcher,
                    dest_user=dest_user,
                    source_user=source_user,
                    config=config
                )
        finally:
            try:
                stids_copier_cache.store()
            except:  # pylint: disable=bare-except
                # ignore this exception, cause it's not our main job
                log.exception('Got exception while try store last cache chunk')
