import logging
import datetime
import pytz

from ora2pg.sharpei import (
    init_in_sharpei,
    get_shard_id,
    get_shard_info,
    get_connstring_by_id
)
from mail.pypg.pypg.common import get_connection
from pymdb.queries import Queries
from pymdb.operations import PurgeUser, PurgeContactsUser
from pymdb.types import Folder

log = logging.getLogger(__name__)


MAX_DEST_USER_AGE = datetime.timedelta(days=3)


class NotSupportedUserError(Exception):
    pass


class SourceUserShouldBeRegistered(NotSupportedUserError):
    pass


class DestUserShouldNotBeRegistered(NotSupportedUserError):
    pass


def register_dest_user_in_pg(dest_user, config):
    init_in_sharpei(
        uid=dest_user.uid,
        dsn=config.sharddb,
        allow_inited=True,
        shard_id=config.dest_shard_id,
    )


def purge_dest_user(dest_user, config):
    with get_connection(get_connstring_by_id(
        sharpei=config.sharpei,
        shard_id=config.dest_shard_id,
        dsn_suffix=config.maildb_dsn_suffix
    )) as conn:
        PurgeUser(conn, dest_user.uid)()
        PurgeContactsUser.by_uid(conn, dest_user.uid)(force=True)


def assert_can_clone_source_user(source_user, config):
    if get_shard_info(source_user.uid, dsn=config.sharddb).shard_id is None:
        raise SourceUserShouldBeRegistered(
            'Got strange pg source_user {0} '
            'he isn\'t initialized in sharpei'.format(source_user))


def is_initialized_in_mdb_for_more_than_1min(uid, shard_id, config):
    with get_connection(get_connstring_by_id(
        sharpei=config.sharpei,
        shard_id=shard_id,
        dsn_suffix=config.maildb_dsn_suffix
    )) as conn:
        folders = Queries(conn, uid).folders()
        for f in folders:
            if f.type == Folder.INBOX:
                return f.created < datetime.datetime.now(pytz.utc) - MAX_DEST_USER_AGE
        return False


def assert_can_clone_into_dest_user(dest_user, config):
    user_shard_id = get_shard_id(dest_user.uid, dsn=config.sharddb)
    if user_shard_id is not None:
        if is_initialized_in_mdb_for_more_than_1min(
                uid=dest_user.uid,
                shard_id=user_shard_id,
                config=config):
            raise DestUserShouldNotBeRegistered(
                'Postgre dest user should not be initialized '
                ' in mdb for more then {age}, uid {uid} shard_id: {shard_id}'.format(
                    age=MAX_DEST_USER_AGE,
                    uid=dest_user.uid,
                    shard_id=user_shard_id
                )
            )
