from abc import ABCMeta, abstractmethod
from logging import getLogger

import pymdb.common as pg
from pymdb.types.db_enums.subscription_state import SubscriptionState
from .pg_get import get_user
from .pg_put import put_user, write_transfer_info

log = getLogger(__name__)


SYSTEM_WORKER_ID = 'husky'


class CopyUserData(object):
    def __init__(self, from_conn, to_dsn, transfer_info, min_received_date=None, max_received_date=None, **options):
        self.from_conn = from_conn
        self.to_dsn = to_dsn
        self.transfer_info = transfer_info
        self.options = options
        self.min_received_date = min_received_date
        self.max_received_date = max_received_date

    @property
    def fill_changelog(self):
        return self.options.get('fill_changelog', False)

    @property
    def tabs_mapping(self):
        return self.options.get('tabs_mapping')

    @property
    def with_presync(self):
        return self.options.get('with_presync', False)


class CopyUser(CopyUserData):
    __metaclass__ = ABCMeta

    @abstractmethod
    def load(self):
        pass

    @abstractmethod
    def to_conn(self):
        pass

    @abstractmethod
    def save(self, to_conn, user):
        pass

    @staticmethod
    def customize_from_user(from_user):
        return from_user

    def customize_to_user(self, to_user):
        return to_user

    @staticmethod
    def convert(to_conn, user):
        return user, None

    def copy(self):
        with self.to_conn() as to_conn:
            from_user = self.customize_from_user(self.load())
            user, mapper = self.convert(to_conn=to_conn, user=from_user)
            self.save(to_conn=to_conn, user=self.customize_to_user(user))
            return mapper


class FromPgMixin(CopyUserData):
    def load(self):
        from_uid = self.transfer_info.src.uid
        assert from_uid, 'requires from_uid'

        pg_user = get_user(
            uid=from_uid,
            conn=self.from_conn,
            min_received_date=self.min_received_date,
            max_received_date=self.max_received_date,
            with_presync=self.with_presync,
        )
        if not pg_user:
            raise RuntimeError('Cannot get data from uid={0}'.format(from_uid))
        return pg_user


class ToPgMixin(CopyUserData):
    def to_conn(self):
        return pg.transaction(self.to_dsn)

    def save(self, to_conn, user):
        to_uid = self.transfer_info.dst.uid
        assert to_uid, 'requires to_uid'

        put_user(
            user, to_uid, conn=to_conn,
            fill_changelog=self.fill_changelog,
            tabs_mapping=self.tabs_mapping,
        )
        write_transfer_info(to_conn, self.transfer_info, to_uid)


def cut_firstline(line, target_len=80, max_word_len=20):
    if not line or len(line) <= target_len:
        return line
    space_ind = line.find(' ', target_len)
    if space_ind == -1:
        return line[:min(len(line), target_len + max_word_len)]
    else:
        return line[:space_ind]


class Pg2Pg(FromPgMixin, ToPgMixin, CopyUser):
    def customize_to_user(self, to_user):
        to_user.shared_folder_subscriptions = [
            sub for sub in to_user.shared_folder_subscriptions
            if SubscriptionState(sub.state) != SubscriptionState.terminated]
        for sub in to_user.shared_folder_subscriptions:
            if SubscriptionState(sub.state) != SubscriptionState.migrate_finished:
                raise RuntimeError('Cannot transfer subscription {0} with state {1}'.format(
                    sub.subscription_id, sub.state))
            sub.worker_id = SYSTEM_WORKER_ID
            sub.state = SubscriptionState.sync
        fl_opts = self.options.get('firstline_options')
        if fl_opts and fl_opts.cut_fl:
            to_user.mails = self.cut_fl_mails(fl_opts, to_user.mails)
        return to_user

    @staticmethod
    def cut_fl_mails(fl_opts, mails):
        for mail in mails:
            max_date = fl_opts.max_date.replace(tzinfo=mail.coords.received_date.tzinfo)
            if mail.coords.received_date <= max_date:
                mail.headers.firstline = cut_firstline(mail.headers.firstline, fl_opts.target_len)
            yield mail


def pg2pg(from_pg_conn, to_pg_dsn, transfer_info,
          fill_changelog=False, tabs_mapping=None, firstline_options=None, min_received_date=None,
          Copier=Pg2Pg):
    return Copier(
        from_conn=from_pg_conn,
        to_dsn=to_pg_dsn,
        transfer_info=transfer_info,
        tabs_mapping=tabs_mapping,
        fill_changelog=fill_changelog,
        firstline_options=firstline_options,
        min_received_date=min_received_date,
    ).copy()
