import logging

from ora2pg import copy_user
from ora2pg.sharpei import get_connstring_by_id
from ora2pg.transfer_data import get_transfer_info, get_user_in_endpoint
from ora2pg.transfer import InvalidTransferArguments, UserAlreadyTransfered
from mail.pypg.pypg.common import transaction

log = logging.getLogger(__name__)


def mark_user_as_moved_from_here_in_future(dsn, uid):
    with transaction(dsn) as conn:
        cur = conn.cursor()
        cur.execute(
            '''UPDATE mail.users
                  SET is_here=false,
                      purge_date=current_timestamp + interval '1 month'
                WHERE uid=%(uid)s
                  AND is_here''',
            dict(uid=uid)
        )


def make_transfer_info(user, to_db, from_db):
    return get_transfer_info(
        src=get_user_in_endpoint(
            db_endpoint=from_db,
            uid=user.uid,
        ),
        dst=get_user_in_endpoint(
            db_endpoint=to_db,
            uid=user.uid,
        ),
    )


def duplicate_user(app, user, to_db, from_db):
    log.debug('Duplicate %r to %r', user, to_db)

    if not to_db.shard_id or not from_db.shard_id:
        raise InvalidTransferArguments('Duplicate requires you to specify source and destination shard_id')

    from_shard_id = from_db.shard_id
    to_shard_id = to_db.shard_id

    if from_shard_id == to_shard_id:
        raise UserAlreadyTransfered('User is already in shard_id={0}'.format(from_shard_id))

    transfer_info = make_transfer_info(user, to_db, from_db)

    from_pg_dsn = get_connstring_by_id(
        sharpei=app.args.sharpei,
        shard_id=from_shard_id,
        dsn_suffix=app.args.maildb_dsn_suffix,
    )
    to_pg_dsn = get_connstring_by_id(
        sharpei=app.args.sharpei,
        shard_id=to_shard_id,
        dsn_suffix=app.args.maildb_dsn_suffix,
    )
    if from_pg_dsn == to_pg_dsn:
        raise UserAlreadyTransfered('User is already in this database, dsn: {0}'.format(from_pg_dsn))

    with transaction(from_pg_dsn) as from_pg_conn:
        copy_user.pg2pg(
            from_pg_conn=from_pg_conn,
            to_pg_dsn=to_pg_dsn,
            transfer_info=transfer_info,
        )
        mark_user_as_moved_from_here_in_future(to_pg_dsn, user.uid)

    log.info('Duplicate is complete.')
