from lxml import etree as ET
from six import text_type

from pymdb.types import MailMimePart


class MimeXmlError(Exception):
    pass


class MimeXmlParseError(MimeXmlError):
    pass


class ConvertError(MimeXmlError):
    pass


class MissedArgumentError(MimeXmlError):
    pass


class MimeXmlNoPartsError(MimeXmlError):
    pass


def str_to_positive_int(name, value):
    try:
        res = int(value)
        if (res < 0):
            raise ConvertError("%s field can't be negative" % name)
        return res
    except ValueError as error:
        raise ConvertError('Invalid %s field format: %s' % (name, error))


class MimePartConvertor(object):
    __slots__ = (
        'hid', 'content_type', 'content_subtype', 'boundary', 'name',
        'charset', 'encoding', 'content_disposition', 'filename', 'cid',
        'offset_begin', 'offset_end')

    _mandatory = ('id', 'offset', 'length')

    _nc = {
        'id': 'hid',
        'content_type.type': 'content_type',
        'content_type.subtype': 'content_subtype',
        'content_type.name': 'name',
        'content_type.charset': 'charset',
        'content_transfer_encoding': 'encoding',
        'content_disposition.value': 'content_disposition',
        'content_disposition.filename': 'filename',
        'content_id': 'cid',
    }

    def __init__(self, **kwargs):
        for k in self._mandatory:
            if k not in kwargs:
                raise MissedArgumentError("Can't find %s field" % k)

        for k in self.__slots__:
            setattr(self, k, '')

        for k, v in self._nc.items():
            if k in kwargs:
                setattr(self, v, kwargs[k])

        self.offset_begin = str_to_positive_int('offset', kwargs['offset'])
        self.offset_end = self.offset_begin + str_to_positive_int('length', kwargs['length'])

    def as_mail_mime_part(self):
        return MailMimePart(**dict((k, getattr(self, k)) for k in self.__slots__))


def parse_mime_xml(data):
    if not data:
        return None
    if isinstance(data, text_type):
        data = data.encode('utf-8')
    try:
        tree = ET.XML(data)
    except SyntaxError as exc:
        raise MimeXmlParseError(
            'Malformed mime xml: %s' % exc
        )
    mime = [MimePartConvertor(**part.attrib).as_mail_mime_part() for part in tree.findall('.//part')]
    if mime == []:
        raise MimeXmlNoPartsError('No parts in mime xml')
    return mime
