from functools import partial

from ora2pg.pg_types import PgUser

from mail.pypg.pypg.common import get_connection
from pymdb.queries import Queries
from pymdb.types import register_types
from ora2pg.errors import EmptyUserError

import logging

log = logging.getLogger(__name__)


def disable_pop3(mails):
    for m in mails:
        m.pop3 = False
        yield m


def user_field_fetcher(q, fieldname, min_received_date, max_received_date):
    query_names_by_user_field = {
        'mails': partial(q.lazy_mails, min_received_date=min_received_date, max_received_date=max_received_date),
        'deleted_mails': partial(q.deleted_mails, min_received_date=min_received_date, max_received_date=max_received_date),
        'synced_messages': partial(q.lazy_synced_messages, min_received_date=min_received_date, max_received_date=max_received_date),
        'message_references': partial(q.message_references, min_received_date=min_received_date, max_received_date=max_received_date),
        'threads_hashes': partial(q.threads_hashes, min_received_date=min_received_date, max_received_date=max_received_date),
        'windat_messages': partial(q.windat_messages, min_received_date=min_received_date, max_received_date=max_received_date),
        'mailish_messages': partial(q.mailish_messages, min_received_date=min_received_date, max_received_date=max_received_date),
    }
    return query_names_by_user_field.get(fieldname) or q.lazy(fieldname)


def user_field_fetcher_with_presync(q, fieldname, min_received_date, max_received_date):
    query_names_by_user_field = {
        'mails': partial(q.lazy_mails, min_received_date=min_received_date, max_received_date=max_received_date),
        'deleted_mails': q.deleted_mails,
        'synced_messages': partial(q.lazy_synced_messages, min_received_date=min_received_date, max_received_date=max_received_date),
        'message_references': q.message_references,
        'threads_hashes': q.threads_hashes,
        'windat_messages': q.windat_messages,
        'mailish_messages': q.mailish_messages,
    }
    return query_names_by_user_field.get(fieldname) or q.lazy(fieldname)


def get_user(uid, dsn=None, conn=None, min_received_date=None, max_received_date=None, with_presync=False):
    with get_connection(dsn, conn) as conn:
        register_types(conn)
        log.info("start reading uid=%s", uid)
        user = PgUser()
        q = Queries(conn, uid)

        for table in PgUser.__slots__:
            # Like user.folders = q.lazy('folders') and so on
            if with_presync:
                fetcher = user_field_fetcher_with_presync(q, table, min_received_date=min_received_date, max_received_date=max_received_date)
            else:
                fetcher = user_field_fetcher(q, table, min_received_date=min_received_date, max_received_date=max_received_date)
            if fetcher:
                setattr(user, table, fetcher())
            else:
                log.warning('%s was not read from PG for uid=%s', table, uid)

        if not user.folders:
            raise EmptyUserError(
                'No folders found for pg user with uid=%r' % uid)

        return user
