import json

from datetime import datetime
from json import dumps
from psycopg2 import IntegrityError

from ora2pg.tools import ImapPager
from pymdb.common import Q as COMMON_Q
from pymdb import types

from pymdb.tools import strip_q
from mail.pypg.pypg.common import simple_executemany, get_connection, exec_simple_insert, qexec
from mail.pypg.pypg.query_conf import load_from_package

import logging

from ora2pg.tools.tabs import (
    LidTabMap,
    make_lid_tab_map,
    mark_user_as_can_read_tabs,
)

log = logging.getLogger(__name__)
Q = load_from_package(__package__, __file__)


class UserAlreadyExists(RuntimeError):
    pass


class StoreUser(object):    # pylint: disable=R0902
    def __init__(self, conn, uid):
        self.conn = conn
        self.uid = uid

    def __repr__(self):
        return 'StoreUser(uid={0.uid})'.format(self)

    def _callproc(self, procedure):
        query = 'SELECT {0}(%(uid)s)'.format(procedure)
        self.conn.cursor().execute(
            query,
            dict(uid=self.uid)
        )

    def base_init(self, data_version, can_read_tabs, is_deleted, state, last_state_update, notifies_count):
        cur = self.conn.cursor()
        mail_user_args = dict(
            uid=self.uid,
            is_here=True,
            is_deleted=is_deleted,
            here_since=datetime.now(),
            can_read_tabs=can_read_tabs,
            data_version=data_version,
            state=state,
            last_state_update=last_state_update,
            notifies_count=notifies_count,
        )

        try:
            exec_simple_insert(cur, "mail.users", **mail_user_args)
        except IntegrityError as exc:
            raise UserAlreadyExists('user: {0} already exists: {1}'.format(
                self.uid, exc))

    def set_user_is_here(self):
        self.conn.cursor().execute(
            strip_q(
                """
                UPDATE mail.users
                   SET is_here = true,
                       purge_date = NULL
                 WHERE uid = %(uid)s
                """
            ),
            dict(
                uid=self.uid,
            )
        )

    def _simple_write(self, objects, cls, table_name):
        write_data = (
            dict(
                uid=self.uid,
                **o.as_dict()
            ) for o in objects
        )
        cur = self.conn.cursor()
        return simple_executemany(cur, table_name, write_data, cls.__slots__ + ('uid',))

    def _simple_write_and_log(self, objects, cls, table_name):
        cnt = self._simple_write(objects, cls, table_name)
        log.info('Create %d records into %s table', cnt, table_name)

    def _passport_user_write(self, objects, cls, table_name):
        write_data = (
            dict(
                user_id=self.uid,
                user_type='passport_user',
                **o.as_dict()
            ) for o in objects
        )
        cur = self.conn.cursor()
        return simple_executemany(cur, table_name, write_data, cls.__slots__ + ('user_id', 'user_type'))

    def write_folders(self, folders):
        self._simple_write(folders, types.Folder, 'mail.folders')
        log.info('Create %d folders', len(folders))

    def update_folders(self, folders):
        for folder in folders:
            qexec(
                self.conn,
                Q.update_folders,
                uid=self.uid,
                fid=folder.fid,
                revision=folder.revision,
                name=folder.name,
                parent_fid=folder.parent_fid,
                type=folder.type,
                unvisited=folder.unvisited,
                unique_type=folder.unique_type,
                created=folder.created,
                next_imap_id=folder.next_imap_id,
                uidvalidity=folder.uidvalidity,
                first_unseen=folder.first_unseen,
                first_unseen_id=folder.first_unseen_id,
                message_count=folder.message_count,
                message_seen=folder.message_seen,
                message_size=folder.message_size,
                message_recent=folder.message_recent,
                attach_count=folder.attach_count,
                attach_size=folder.attach_size,
                pop3state=folder.pop3state,
                position=folder.position,
                subscribed_for_shared_folder=folder.subscribed_for_shared_folder,
            )
        log.info('Update %d folders', len(folders))

    def write_tabs(self, tabs):
        self._simple_write(tabs, types.Tab, 'mail.tabs')
        log.info('Create %d tabs', len(tabs))

    def write_shared_folders(self, shared_folders):
        self._simple_write(shared_folders, types.SharedFolder, 'mail.shared_folders')
        log.info('Create %d shared folders', len(shared_folders))

    def write_shared_folder_subscriptions(self, shared_folder_subscriptions):
        self._simple_write(shared_folder_subscriptions,
                           types.SharedFolderSubscription, 'mail.shared_folder_subscriptions')
        log.info('Create %d shared folder subscriptions', len(shared_folder_subscriptions))

    def write_subscribed_folders(self, subscribed_folders):
        self._simple_write(subscribed_folders, types.SubscribedFolder, 'mail.subscribed_folders')
        log.info('Create %d subscribed folders', len(subscribed_folders))

    def update_subscribed_folders(self, subscribed_folders):
        for subscribed_folder in subscribed_folders:
            qexec(
                self.conn,
                Q.update_subscribed_folders,
                uid=self.uid,
                fid=subscribed_folder.fid,
                revision=subscribed_folder.revision,
                owner_uid=subscribed_folder.owner_uid,
                owner_fid=subscribed_folder.owner_fid,
                subscription_id=subscribed_folder.subscription_id,
                synced_revision=subscribed_folder.synced_revision,
                created=subscribed_folder.created,
                synced_imap_id=subscribed_folder.synced_imap_id,
            )
        log.info('Update %d subscribed folders', len(subscribed_folders))

    def write_synced_messages(self, synced_messages):
        cnt = self._simple_write(synced_messages, types.SyncedMessage, 'mail.synced_messages')
        log.info('Create %d synced messages', cnt)

    def write_folder_archivation_rules(self, folder_archivation_rules):
        self._simple_write(folder_archivation_rules, types.FolderArchivationRule, 'mail.folder_archivation_rules')
        log.info('Create %d folder archivation rules', len(folder_archivation_rules))

    def write_imap_unsubscribed_folders(self, imap_unsubscribed_folders):
        self._simple_write(
            imap_unsubscribed_folders,
            types.IMAPUnsubscribedFolder,
            'mail.imap_unsubscribed_folders')

    def write_labels(self, labels):
        self._simple_write(
            labels,
            types.Label,
            'mail.labels')
        log.info("Create %d labels", len(labels))

    def write_windat_messages(self, windat_messages):
        cnt = self._simple_write(windat_messages, types.WinDatMessages, 'mail.windat_messages')
        log.info('Create %d windat_messages', cnt)

    def write_settings(self, settings):
        for setting in settings:
            setting.value = dumps(setting.value)
        self._simple_write(settings, types.Settings, 'settings.settings')
        log.info('Create %d settings', len(settings))

    def write_user_stats(self, user_stats):
        self._simple_write(user_stats, types.UserStats, 'stats.users_info')

    def write_archives(self, archives):
        self._simple_write(archives, types.Archives, 'mail.archives')

    def write_stickers_reply_later(self, stickers):
        self._simple_write(stickers, types.StickersReplyLater, 'mail.stickers_reply_later')

    def write_backups_info(self, user):
        self._simple_write_and_log(user.backups, types.Backup, 'backup.backups')
        self._simple_write_and_log(user.restores, types.Restore, 'backup.restores')
        self._simple_write_and_log(user.backup_folders, types.BackupFolders, 'backup.folders')
        self._simple_write_and_log(user.backup_box, types.BackupBox, 'backup.box')
        self._simple_write_and_log(user.folders_to_backup, types.FoldersToBackup, 'backup.folders_to_backup')
        self._simple_write_and_log(user.tabs_to_backup, types.TabsToBackup, 'backup.tabs_to_backup')

    def write_contacts(self, user):
        self.write_contacts_user(user.passport_user_contacts_users)
        self.write_contacts_serials(user.passport_user_contacts_serials)
        self.write_contacts_lists(user.passport_user_contacts_lists)
        self.write_contacts_tags(user.passport_user_contacts_tags)
        self.write_contacts_contacts(user.passport_user_contacts_contacts)
        self.write_contacts_emails(user.passport_user_contacts_emails)
        self.write_contacts_contacts_tags(user.passport_user_contacts_contacts_tags)
        self.write_contacts_emails_tags(user.passport_user_contacts_emails_tags)
        self.write_contacts_shared_lists(user.passport_user_contacts_shared_lists)
        self.write_contacts_subscribed_lists(user.passport_user_contacts_subscribed_lists)

    def write_contacts_user(self, contacts_users):
        self._passport_user_write(contacts_users, types.ContactsUser, 'contacts.users')
        log.info('Create %d contacts.users', len(contacts_users))

    def write_contacts_serials(self, contacts_serials):
        self._passport_user_write(contacts_serials, types.ContactsSerials, 'contacts.serials')
        log.info('Create %d contacts.serials', len(contacts_serials))

    def write_contacts_lists(self, contacts_lists):
        self._passport_user_write(contacts_lists, types.ContactsList, 'contacts.lists')
        log.info('Create %d contacts.lists', len(contacts_lists))

    def write_contacts_tags(self, contacts_tags):
        self._passport_user_write(contacts_tags, types.ContactsTag, 'contacts.tags')
        log.info('Create %d contacts.tags', len(contacts_tags))

    def write_contacts_contacts(self, contacts_contacts):
        for contact in contacts_contacts:
            contact.vcard = json.dumps(contact.vcard)
        self._passport_user_write(contacts_contacts, types.Contact, 'contacts.contacts')
        log.info('Create %d contacts.contacts', len(contacts_contacts))

    def write_contacts_emails(self, contacts_emails):
        self._passport_user_write(contacts_emails, types.ContactsEmail, 'contacts.emails')
        log.info('Create %d contacts.emails', len(contacts_emails))

    def write_contacts_contacts_tags(self, contacts_contacts_tags):
        self._passport_user_write(contacts_contacts_tags, types.ContactTag, 'contacts.contacts_tags')
        log.info('Create %d contacts.contacts_tags', len(contacts_contacts_tags))

    def write_contacts_emails_tags(self, contacts_emails_tags):
        self._passport_user_write(contacts_emails_tags, types.ContactsEmailTag, 'contacts.emails_tags')
        log.info('Create %d contacts.emails_tags', len(contacts_emails_tags))

    def write_contacts_shared_lists(self, contacts_shared_lists):
        self._passport_user_write(contacts_shared_lists, types.SharedContactsListTo, 'contacts.shared_lists')
        log.info('Create %d contacts.shared_lists', len(contacts_shared_lists))

    def write_contacts_subscribed_lists(self, contacts_subscribed_lists):
        self._passport_user_write(contacts_subscribed_lists, types.SubscribedContactsListTo, 'contacts.subscribed_lists')
        log.info("Create %d contacts.subscribed_lists", len(contacts_subscribed_lists))

    def write_collectors(self, collectors):
        for collector in collectors:
            collector.metadata = dumps(collector.metadata)
        self._simple_write(collectors, types.Collector, 'mail.collectors')
        log.info('Create %d collectors', len(collectors))

    def add_mails(self, mails, imap_pager, lid_tab_map=LidTabMap(), inbox_fid=None):    # pylint: disable=R0914
        def split_mail(m):
            if m.coords.tab is None and m.coords.fid == inbox_fid:
                tab = lid_tab_map.get_tab(m.lids)
            else:
                tab = m.coords.tab
            return (
                dict(
                    uid=self.uid,
                    mid=m.mid,
                    fid=m.coords.fid,
                    tid=m.coords.tid,
                    imap_id=imap_pager(m),
                    revision=m.coords.revision,
                    chain=None,
                    seen=m.coords.seen,
                    recent=m.coords.recent,
                    deleted=m.coords.deleted,
                    received_date=m.coords.received_date,
                    lids=m.lids,
                    doom_date=m.doom_date,
                    tab=tab,
                    newest_tit=(None if tab is None else False),
                ),
                dict(
                    uid=self.uid,
                    mid=m.mid,
                    size=m.coords.size,
                    st_id=m.coords.st_id,
                    attaches=m.attaches,
                    mime=m.mime,
                    attributes=types.ListOfMessageAttributes(m.coords.attributes),
                    subject=m.headers.subject,
                    firstline=m.headers.firstline,
                    hdr_date=m.headers.hdr_date,
                    hdr_message_id=m.headers.hdr_message_id,
                    recipients=m.recipients,
                    extra_data=m.headers.extra_data,
                    pop_uidl=m.coords.pop_uidl,
                    found_tid=m.coords.found_tid,
                    thread_rule=m.coords.thread_rule,
                ),
                dict(
                    uid=self.uid,
                    mid=m.mid,
                    fid=m.coords.fid,
                    size=m.coords.size,
                    old_uidl=m.pop_uidl) if m.pop3 else None
            )

        def split_box_messages_pop3(mails):
            from mail.pypg.pypg.common import chunks

            for chunk in chunks(mails):
                box = []
                messages = []
                pop3_messages = []
                for m in chunk:
                    box_item, message_item, pop3_item = split_mail(m)
                    box.append(box_item)
                    messages.append(message_item)
                    if pop3_item:
                        pop3_messages.append(pop3_item)
                yield box, messages, pop3_messages

        cur = self.conn.cursor()

        msg_count = 0
        pop3_count = 0
        for mailbox_meta, messages_meta, pop3_meta in split_box_messages_pop3(mails):
            simple_executemany(cur, 'mail.box', mailbox_meta)
            simple_executemany(cur, 'mail.messages', messages_meta)
            simple_executemany(cur, 'mail.pop3_box', pop3_meta)
            msg_count += len(messages_meta)
            pop3_count += len(pop3_meta)

        log.info("store messages: {0}".format(msg_count))
        log.info("store pop3 messages: {0}".format(pop3_count))

    def fill_imap_chains(self):
        log.info('fill imap chanins')
        self._callproc('util.fill_imap_chains')

    def fill_aggreate_counters(self):
        log.info("build counters and threads structures")
        self.conn.cursor().execute(
            """DELETE FROM mail.threads WHERE uid = %(uid)s""",
            dict(uid=self.uid)
        )
        self._callproc('util.fill_counters')

    def fill_threads(self, message_references, threads_hashes):

        log.debug("creating refs_meta and threads_hashes")
        cur = self.conn.cursor()

        for table_name, meta, PgType in [
                ('message_references', message_references, types.MessageReference),
                ('threads_hashes', threads_hashes, types.ThreadsHash)]:
            log.info("store into {0}".format(table_name))
            cnt = simple_executemany(
                cur,
                'mail.{0}'.format(table_name),
                (dict(uid=self.uid, **m.as_dict()) for m in meta if m),
                PgType.__slots__ + ('uid',)
            )
            log.info('Store %d into %s', cnt, table_name)

    def fill_counters(self, counters, attach_counters):
        log.info("setting mail.counters for {0} to {1} and {2}".format(self.uid, counters, attach_counters))
        cur = self.conn.cursor()
        cur.execute(strip_q(
            """INSERT INTO mail.counters AS c
                (uid, fresh_count, revision,
                 has_attaches_count, has_attaches_seen)
               VALUES
                (%s, %s, %s, %s, %s)
               ON CONFLICT (uid) DO UPDATE SET
                 fresh_count = excluded.fresh_count,
                 revision = greatest(c.revision, excluded.revision),
                 has_attaches_count = excluded.has_attaches_count,
                 has_attaches_seen = excluded.has_attaches_seen"""),
            (self.uid, counters.fresh_count, counters.revision,
             attach_counters.has_attaches_count,
             attach_counters.has_attaches_seen)
        )

    def write_filters(self, filters):
        cur = self.conn.cursor()

        log.info('Generating {0} ids for rules'.format(len(filters)))
        cur.execute(strip_q("""
            SELECT nextval('filters.rule_seq') as new_id
              FROM generate_series(1, %s)
            """), (len(filters),))
        for r in iter(filters):
            r.rule_id = cur.fetchone()[0]

        rules=(
            dict(
                uid=self.uid,
                **r.as_dict()
            ) for r in iter(filters)
        )
        simple_executemany(cur, 'filters.rules', rules,
                           ['uid', 'rule_id', 'name', 'enabled', 'prio', 'stop', 'created', 'type'])

        from itertools import chain, repeat
        flat_acts = [
            (a, id) for a, id in chain.from_iterable(
                zip(iter(r.actions), repeat(r.rule_id))
                for r in iter(filters))
        ]
        verifiable_action = lambda a : a.oper in ['forward', 'forwardwithstore', 'notify']
        verified_params = set(
            a.param for a, rule_id in flat_acts
            if verifiable_action(a) and a.verified is True)

        log.info('Generating {0} ids for actions'.format(len(flat_acts)))
        cur.execute(strip_q("""
            SELECT nextval('filters.action_seq') as new_id
              FROM generate_series(1, %s)
            """), (len(flat_acts),))
        actions=[
            dict(
                uid=self.uid,
                rule_id=id,
                action_id=cur.fetchone()[0],
                oper=a.oper,
                param=a.param,
                verified=True if a.verified
                    or (verifiable_action(a) and a.param in verified_params)
                    else False,
            ) for a, id in flat_acts]
        simple_executemany(cur, 'filters.actions', actions,
                           ['uid', 'rule_id', 'action_id', 'oper', 'param', 'verified'])

        flat_conds = chain.from_iterable(
            zip(iter(r.conditions), repeat(r.rule_id))
            for r in iter(filters)
        )
        conditions=(
            dict(
                uid=self.uid,
                rule_id=id,
                **c.as_dict()
            ) for c, id in flat_conds)
        simple_executemany(cur, 'filters.conditions', conditions,
                           ['uid', 'rule_id', 'field_type', 'field', 'pattern', 'oper', 'link', 'negative'])
        log.info('Store {0} filter rules'.format(len(filters)))

    def write_filter_elists(self, filter_elists):
        simple_executemany(
            self.conn.cursor(),
            'filters.elist',
            (dict(uid=self.uid, **e.as_dict()) for e in filter_elists),
            ['uid', 'email', 'list', 'created'],
        )

    def write_serials(self, serials):
        log.info("writer serials %r", serials)
        self.conn.cursor().execute(strip_q(
            """INSERT INTO mail.serials AS s
                (uid, next_revision, next_fid, next_lid, next_mid_serial,
                 next_owner_subscription_id, next_subscriber_subscription_id, next_collector_id, next_backup_id)
               VALUES
                (%s, %s, %s, %s, %s, %s, %s, %s, %s)
               ON CONFLICT (uid) DO UPDATE SET
                 next_revision = greatest(s.next_revision, excluded.next_revision),
                 next_fid = greatest(s.next_fid, excluded.next_fid),
                 next_lid = greatest(s.next_lid, excluded.next_lid),
                 next_mid_serial = greatest(s.next_mid_serial, excluded.next_mid_serial),
                 next_owner_subscription_id = greatest(s.next_owner_subscription_id, excluded.next_owner_subscription_id),
                 next_subscriber_subscription_id = greatest(s.next_subscriber_subscription_id, excluded.next_subscriber_subscription_id),
                 next_collector_id = greatest(s.next_collector_id, excluded.next_collector_id),
                 next_backup_id = greatest(s.next_backup_id, excluded.next_backup_id)
                 """),
            (self.uid, serials.next_revision, serials.next_fid, serials.next_lid, serials.next_mid_serial,
             serials.next_owner_subscription_id, serials.next_subscriber_subscription_id, serials.next_collector_id, serials.next_backup_id)
        )

    def add_uid_escape_json(self, rows, json_value_name):
        def do_it(row):
            retval = dict(uid=self.uid, **row.as_dict())
            retval[json_value_name] = dumps(retval[json_value_name])
            return retval

        return (do_it(r) for r in rows)

    def write_fix_log(self, fix_log):
        simple_executemany(
            self.conn.cursor(),
            'mail.fix_log',
            self.add_uid_escape_json(fix_log, 'fixed'),
            ['uid'] + list(types.FixLog.__slots__)
        )

    def write_deleted_box(self, deleted_mails):
        def split_mail(m):
            return (
                dict(
                    uid=self.uid,
                    mid=m.mid,
                    deleted_date=m.coords.deleted_date,
                    revision=m.coords.revision,
                    info=dumps(m.coords.info),
                    received_date=m.coords.received_date,
                ),
                dict(
                    uid=self.uid,
                    mid=m.mid,
                    size=m.coords.size,
                    st_id=m.coords.st_id,
                    attaches=m.attaches,
                    mime=m.mime,
                    attributes=types.ListOfMessageAttributes(m.coords.attributes),
                    subject=m.headers.subject,
                    firstline=m.headers.firstline,
                    hdr_date=m.headers.hdr_date,
                    hdr_message_id=m.headers.hdr_message_id,
                    recipients=m.recipients,
                    extra_data=m.headers.extra_data,
                    pop_uidl=m.coords.pop_uidl,
                    found_tid=m.coords.found_tid,
                    thread_rule=m.coords.thread_rule,
                ),
            )

        def split_deleted_box_messages(mails):
            from mail.pypg.pypg.common import chunks

            for chunk in chunks(mails):
                deleted_box = []
                messages = []
                for m in chunk:
                    deleted_box_item, message_item = split_mail(m)
                    deleted_box.append(deleted_box_item)
                    messages.append(message_item)
                yield deleted_box, messages

        cur = self.conn.cursor()
        msg_count = 0

        for deleted_box_meta, messages_meta, in split_deleted_box_messages(deleted_mails):
            simple_executemany(cur, 'mail.messages', messages_meta)
            simple_executemany(cur, 'mail.deleted_box', deleted_box_meta)
            msg_count += len(messages_meta)
        log.info("store deleted messages: {0}".format(msg_count))

    def fill_change_log(self):
        log.info('Fill change log with copied messages')
        self._callproc('util.fill_changelog_for_msearch')

    def apply_data_migrations(self):
        self._callproc('util.apply_data_migrations')

    def write_mailish_accounts(self, mailish_accounts):
        cnt = self._simple_write(mailish_accounts, types.MailishAccount, 'mailish.accounts')
        log.info('Create %d mailish accounts', cnt)

    def write_mailish_auth_data(self, mailish_auth_data):
        cnt = self._simple_write(mailish_auth_data, types.MailishAuthData, 'mailish.auth_data')
        log.info('Create %d mailish auth data', cnt)

    def write_mailish_folders(self, mailish_folders):
        cnt = self._simple_write(mailish_folders, types.MailishFolder, 'mailish.folders')
        log.info('Create %d mailish folders', cnt)

    def write_mailish_messages(self, mailish_messages):
        cnt = self._simple_write(mailish_messages, types.MailishMessage, 'mailish.messages')
        log.info('Create %d mailish messages', cnt)

    @staticmethod
    def get_inbox_fid(folders):
        for f in folders:
            if f.type == 'inbox':
                return f.fid
        return None

    def set_tab_flags(self, tabs_mapping):
        if tabs_mapping is not None:
            mark_user_as_can_read_tabs(self.conn, self.uid)
            log.info('Set can_read_tabs to true')

    def put(self, user, fill_changelog, tabs_mapping=None):
        self.fill_counters(user.counters, user.attach_counters)
        if user.mailish_accounts:
            self.write_mailish_accounts(user.mailish_accounts)
        if user.mailish_auth_data:
            self.write_mailish_auth_data(user.mailish_auth_data)
        self.write_folders(user.folders)
        self.write_tabs(user.tabs)
        self.write_shared_folders(user.shared_folders)
        self.write_shared_folder_subscriptions(user.shared_folder_subscriptions)
        self.write_subscribed_folders(user.subscribed_folders)
        self.write_folder_archivation_rules(user.folder_archivation_rules)
        self.write_imap_unsubscribed_folders(user.imap_unsubscribed_folders)
        if user.mailish_folders:
            self.write_mailish_folders(user.mailish_folders)
        self.write_labels(user.labels)
        self.write_filters(user.filters)
        self.write_filter_elists(user.filter_elists)
        pager = ImapPager(user.folders)
        self.add_mails(user.mails, pager,
                       lid_tab_map=make_lid_tab_map(user.labels, tabs_mapping),
                       inbox_fid=self.get_inbox_fid(user.folders))
        self.write_synced_messages(user.synced_messages)
        if user.mailish_messages:
            self.write_mailish_messages(user.mailish_messages)
        self.write_serials(user.serials)
        # after serials insert, cause they look at serials
        self.fill_imap_chains()
        self.fill_aggreate_counters()
        # after fill_aggreate_counters,
        # cause util.fill_counters build mail.threads
        # and we have FK threads_hashes -> threads
        self.fill_threads(
            user.message_references,
            user.threads_hashes
        )
        self.write_fix_log(user.fix_log)
        self.write_deleted_box(user.deleted_mails)
        self.write_windat_messages(user.windat_messages)
        self.write_archives(user.archives)
        self.write_stickers_reply_later(user.stickers_reply_later)
        self.write_backups_info(user)
        self.write_settings(user.settings)
        self.write_contacts(user)
        self.write_collectors(user.collectors)
        self.set_tab_flags(tabs_mapping)
        self.apply_data_migrations()
        self.write_user_stats(user.stats_users_info)
        # Do not place any staff after fill_change_log() call
        # because of "poor search people"^W change_log users
        # struggling when we take "cid"s from sequence in
        # fill_change_log call and don't let them see entries
        # in change_log 'till we say "COMMIT".
        if fill_changelog:
            self.fill_change_log()

    def presync(self, user, tabs_mapping=None):
        self.fill_counters(user.counters, user.attach_counters)
        self.write_folders(user.folders)
        self.write_subscribed_folders(user.subscribed_folders)
        pager = ImapPager(user.folders)
        self.add_mails(user.mails, pager,
                       lid_tab_map=make_lid_tab_map(user.labels, tabs_mapping),
                       inbox_fid=self.get_inbox_fid(user.folders))
        self.write_synced_messages(user.synced_messages)
        self.write_serials(user.serials)

    def sync(self, user, tabs_mapping=None):
        self.fill_counters(user.counters, user.attach_counters)
        if user.mailish_accounts:
            self.write_mailish_accounts(user.mailish_accounts)
        if user.mailish_auth_data:
            self.write_mailish_auth_data(user.mailish_auth_data)
        self.update_folders(user.folders)
        self.write_tabs(user.tabs)
        self.write_shared_folders(user.shared_folders)
        self.write_shared_folder_subscriptions(user.shared_folder_subscriptions)
        self.update_subscribed_folders(user.subscribed_folders)
        self.write_folder_archivation_rules(user.folder_archivation_rules)
        self.write_imap_unsubscribed_folders(user.imap_unsubscribed_folders)
        if user.mailish_folders:
            self.write_mailish_folders(user.mailish_folders)
        self.write_labels(user.labels)
        self.write_filters(user.filters)
        self.write_filter_elists(user.filter_elists)
        pager = ImapPager(user.folders)
        self.add_mails(user.mails, pager,
                       lid_tab_map=make_lid_tab_map(user.labels, tabs_mapping),
                       inbox_fid=self.get_inbox_fid(user.folders))
        self.write_synced_messages(user.synced_messages)
        if user.mailish_messages:
            self.write_mailish_messages(user.mailish_messages)
        self.write_serials(user.serials)
        self.fill_imap_chains()
        self.fill_aggreate_counters()
        self.fill_threads(
            user.message_references,
            user.threads_hashes
        )
        self.write_fix_log(user.fix_log)
        self.write_deleted_box(user.deleted_mails)
        self.write_windat_messages(user.windat_messages)
        self.write_archives(user.archives)
        self.write_stickers_reply_later(user.stickers_reply_later)
        self.write_backups_info(user)
        self.write_settings(user.settings)
        self.write_contacts(user)
        self.write_collectors(user.collectors)
        self.set_tab_flags(tabs_mapping)


def put_user(user, uid, dsn=None, conn=None, fill_changelog=False, tabs_mapping=None):
    with get_connection(dsn, conn) as conn:
        store = StoreUser(
            conn=conn,
            uid=uid,
        )
        log.info("start {0} transfer".format(store))
        store.base_init(
            data_version=user.data_version,
            can_read_tabs=user.can_read_tabs,
            is_deleted=user.is_deleted,
            state=user.state,
            last_state_update=user.last_state_update,
            notifies_count=user.notifies_count,
        )
        store.put(user, fill_changelog, tabs_mapping)


def presync_user(user, uid, dsn=None, conn=None, tabs_mapping=None):
    with get_connection(dsn, conn) as conn:
        store = StoreUser(
            conn=conn,
            uid=uid,
        )
        log.info("start {0} presync".format(store))
        store.base_init(
            data_version=user.data_version,
            can_read_tabs=user.can_read_tabs,
            is_deleted=user.is_deleted,
            state=user.state,
            last_state_update=user.last_state_update,
            notifies_count=user.notifies_count,
        )
        store.presync(user, tabs_mapping)


def sync_user(user, uid, dsn=None, conn=None, tabs_mapping=None):
    with get_connection(dsn, conn) as conn:
        store = StoreUser(
            conn=conn,
            uid=uid,
        )
        log.info("start {0} sync".format(store))
        store.set_user_is_here()
        qexec(conn, COMMON_Q.lock_user, uid=uid)
        store.sync(user, tabs_mapping)


def write_transfer_info(conn, transfer_info, uid, revision=1):
    log.info('write transfer info')
    cur = conn.cursor()

    transfer_info_dict = dict(
        from_db=str(transfer_info.src.db),
        from_uid=transfer_info.src.uid,
        to_db=str(transfer_info.dst.db),
        to_uid=transfer_info.dst.uid,
        transfer_start=transfer_info.started,
        transfer_end=datetime.now(),
        script=transfer_info.script,
        script_revision=transfer_info.script_revision,
        transfered_at=transfer_info.transfered_at
    )

    exec_simple_insert(cur, 'mail.transfer_info', **transfer_info_dict)

    transfer_info_dict.update({
        'uid': uid,
        'mdb': transfer_info.src.db.db,
        'revision': revision,
    })

    cur.execute(
        strip_q("""
        INSERT INTO mail.change_log
            (uid, revision, type,
            arguments,
            fresh_count, useful_new_count)
        SELECT
            %(uid)s, %(revision)s, 'transfer'::mail.change_type,
            json_build_object(
                'mdb', %(mdb)s,
                'from_db', %(from_db)s,
                'to_db', %(to_db)s
            )::jsonb,
            (impl.get_changelog_counters(%(uid)s)).*;"""),
        transfer_info_dict
    )
