#!/usr/bin/env python
# coding: utf-8

from future.standard_library import install_aliases
install_aliases()

import json
from urllib.parse import urlparse, parse_qs

from logging import getLogger
from ora2pg.tools import http

log = getLogger(__name__)


class Error(RuntimeError):
    pass


class GetError(Error):
    pass


class PutError(Error):
    pass


class DeleteError(Error):
    pass


def handle_http_error(exception):
    def impl(function):
        def wrapped(*args, **kwargs):
            from urllib2 import URLError
            try:
                return function(*args, **kwargs)
            except URLError:
                raise exception
        return wrapped
    return impl


class Api(object):
    def __init__(self, transport):
        self.transport = transport

    @handle_http_error(GetError('Get request to reminders failed'))
    def get(self):
        with self.transport.get() as result:
            return make_reminders(json.load(result))

    @handle_http_error(PutError('Put request to reminders failed'))
    def put(self, reminder):
        with self.transport.put(
            reminder_id=reminder.id,
            data=json.dumps(dict(
                name=reminder.name,
                reminderDate=reminder.date,
                channels=dict(callback=dict(url=reminder.callback_url))))
        ) as result:
            log.debug(result.read())

    @handle_http_error(DeleteError('Delete request to reminders failed'))
    def delete(self, reminder_id):
        with self.transport.delete(reminder_id) as result:
            log.debug(result.read())


class Reminder(object):
    __slots__ = (
        'id',
        'client_id',
        'name',
        'date',
        'callback_url',
    )

    def __init__(self, id, client_id, name, date, callback_url):
        self.id = id
        self.client_id = client_id
        self.name = name
        self.date = date
        self.callback_url = callback_url


class HttpTransport(object):
    CLIENT_ID = 'yandex-wmi'

    def __init__(self, host, uid):
        self.host = host
        self.uid = uid

    def get(self):
        return http.request(
            url=http.url_join(
                host=self.host,
                method=self.__base_location,
            ),
            do_retries=True,
        )

    def put(self, reminder_id, data):
        return http.request(
            url=http.url_join(
                host=self.host,
                method=self.__base_location + '/.put',
                args=dict(id=reminder_id),
            ),
            data=data,
            headers={'Content-Type': 'application/json'},
            do_retries=True,
        )

    def delete(self, reminder_id):
        return http.request(
            url=http.url_join(
                host=self.host,
                method=self.__base_location + '/.delete',
                args=dict(id=reminder_id),
            ),
            do_retries=True,
        )

    @property
    def __base_location(self):
        return 'api/v1/{uid}/reminders/{cid}'.format(
            uid=self.uid,
            cid=self.CLIENT_ID,
        )


def get_callback_url_mid(callback_url):
    url = urlparse(callback_url)
    qs_dict = parse_qs(url.query)
    try:
        return int(qs_dict['mid'][0])
    except (KeyError, IndexError, ValueError) as exc:
        log.warning(
            'Got callback_url %r without mid: %s',
            callback_url,
            exc
        )
        return None


def make_reminders(data):
    return (
        Reminder(
            id=v['id'],
            client_id=v['clientId'],
            name=v['name'],
            date=v['reminderDate'],
            callback_url=v['channels']['callback']['url'],
        ) for v in data['result']['reminders']
    )
