# coding: utf-8
'''
wrapper for message storage api
'''

from future.standard_library import install_aliases
from six import text_type

install_aliases()

from urllib.error import HTTPError
from ora2pg.tools.http import request, url_join

MULCAGATE_PORT = 10010
CLOSE_XML_PART_TAG = b'</message>\n'


def get_headers(tvm_ticket):
    if tvm_ticket is None:
        return {}
    else:
        return {"X-Ya-Service-Ticket": tvm_ticket}


class StorageError(Exception):
    pass


class StorageGetError(StorageError):
    pass


class DeletedMessage(StorageGetError):
    pass


class StoragePutError(StorageError):
    pass


class StorageDelError(StorageError):
    pass


def has_xml_part(data):
    if isinstance(data, text_type):
        data = data.encode('utf-8')
    if data.startswith(b'<?xml') and CLOSE_XML_PART_TAG in data:
        return data.find(b'\n<message>') == data.find(b'\n')
    return False


def split_mime_xml_and_message(data):
    if isinstance(data, text_type):
        data = data.encode('utf-8')
    if has_xml_part(data):
        xml_part, _, message = data.partition(CLOSE_XML_PART_TAG)
        return xml_part + CLOSE_XML_PART_TAG, message
    return None, data


def to_ascii(s):
    if isinstance(s, text_type):
        return s.encode('ascii')
    return s


class MulcaGate(object):
    def __init__(self, host, port=None, tvm_ticket=None, mg_namespace='mail',
                 mg_ca_path=None):
        self.host = host
        self.port = port
        self.tvm_ticket = tvm_ticket
        self.ca_path = mg_ca_path
        self.mg_namespace = mg_namespace

    def get_raw_data(self, st_id, **request_kwargs):
        get_args = {'raw': '', 'service': 'maildbtransfer'}
        try:
            with request(
                url_join(
                    host=self.host,
                    port=self.port or MULCAGATE_PORT,
                    method='gate/get/' + st_id,
                    args=get_args),
                do_retries=True,
                skip_retry_codes=[404],
                headers=get_headers(self.tvm_ticket),
                cafile=self.ca_path,
                **request_kwargs
            ) as fd:
                return fd.read()
        except HTTPError as exc:
            if exc.code == 404:
                raise DeletedMessage(
                    'No such message %s: %s' % (st_id, exc.read())
                )
            raise StorageGetError(
                'Got unknown error on message %s: code: %r, %s' % (
                    st_id, exc.code, exc.read()))
        except IOError as exc:
            raise StorageGetError(
                'Got IOError on messages %s: %s' % (st_id, exc)
            )

    def get(self, st_id, **request_kwargs):
        raw_data = self.get_raw_data(st_id, **request_kwargs)
        _, message = split_mime_xml_and_message(raw_data)
        return message

    def put(self, base_id, data, **request_kwargs):
        data_has_xml_part = has_xml_part(data)
        if data_has_xml_part:
            raise StoragePutError("Can't put to MDS message with mime xml")

        put_args = {
            'elliptics' : '1',
            'raw': '',
            'ns': self.mg_namespace,
            'service': 'maildbtransfer'
            }

        try:
            with request(
                url_join(
                    host=self.host,
                    port=self.port or MULCAGATE_PORT,
                    method='gate/put/%s' % base_id,
                    args=sorted(put_args.items()),
                ),
                data=data,
                do_retries=True,
                timeout=30,
                log_post_data=False,
                headers=get_headers(self.tvm_ticket),
                cafile=self.ca_path,
                **request_kwargs
            ) as fd:
                return text_type(fd.read(), encoding='utf-8')
        except HTTPError as exc:
            raise StoragePutError(
                'Put failed with code %r: response: %s' % (
                    exc.code, exc.read()))
        except IOError as exc:
            raise StoragePutError(
                'Put failed with IOError %s' % exc)

    def delete(self, st_id, **request_kwargs):
        del_args = {'service': 'maildbtransfer'}
        try:
            with request(
                url_join(
                    host=self.host,
                    port=self.port or MULCAGATE_PORT,
                    method='gate/del/' + st_id,
                    args=del_args),
                do_retries=True,
                headers=get_headers(self.tvm_ticket),
                cafile=self.ca_path,
                **request_kwargs
            ) as fd:
                return fd.read()
        except HTTPError as exc:
            raise StorageDelError('Del failed with code %r: response: %s' % (exc.code, exc.read()))
        except IOError as exc:
            raise StorageDelError('Del failed with IOError %s' % exc)
