import logging

from ora2pg import copy_user
from ora2pg.sharpei import get_connstring_by_id, update_shard_id, get_shard_id
from ora2pg.transfer import InvalidTransferArguments, UserAlreadyTransfered, verify_user_is_in_from_db, from_user_is_here
from ora2pg.huskydb import write_transfer_info
from ora2pg.pg_put import write_transfer_info as maildb_write_transfer_info
from ora2pg.tools.find_master_helpers import find_huskydb, find_sharddb
from ora2pg.transfer_delete_queue import transfer_delete_queue
from ora2pg.transfer_subscriptions import prepare_for_migrate_subs, update_subs_status
from mail.pypg.pypg.common import transaction
from pymdb.tools import mark_user_as_moved_from_here, lock_contacts_user, mark_contacts_user_as_moved_from_here
from pymdb.common import locked_transaction

from .duplicate import mark_user_as_moved_from_here_in_future, make_transfer_info
from .pg_put import presync_user as pg_presync_user, sync_user as pg_sync_user

log = logging.getLogger(__name__)


class PresyncPg2Pg(copy_user.Pg2Pg):
    def save(self, to_conn, user):
        to_uid = self.transfer_info.dst.uid
        assert to_uid, 'requires to_uid'
        pg_presync_user(user, to_uid, conn=to_conn, tabs_mapping=self.tabs_mapping)

    def customize_to_user(self, to_user):
        return to_user


class SyncPg2Pg(copy_user.Pg2Pg):
    def save(self, to_conn, user):
        to_uid = self.transfer_info.dst.uid
        assert to_uid, 'requires to_uid'
        pg_sync_user(user, to_uid, conn=to_conn, tabs_mapping=self.tabs_mapping)
        maildb_write_transfer_info(to_conn, self.transfer_info, to_uid)


def presync_user(app, user, to_db, from_db, presync_received_date):
    log.debug('Pre-sync %r to %r', user, to_db)

    if not to_db.shard_id or not from_db.shard_id:
        raise InvalidTransferArguments('Pre-sync requires you to specify source and destination shard_id')

    from_shard_id = from_db.shard_id
    to_shard_id = to_db.shard_id

    if from_shard_id == to_shard_id:
        raise UserAlreadyTransfered('User is already in shard_id={0}'.format(from_shard_id))

    transfer_info = make_transfer_info(user, to_db, from_db)

    from_pg_dsn = get_connstring_by_id(app.args.sharpei, from_shard_id, app.args.maildb_dsn_suffix)
    to_pg_dsn = get_connstring_by_id(app.args.sharpei, to_shard_id, app.args.maildb_dsn_suffix)

    if from_pg_dsn == to_pg_dsn:
        raise UserAlreadyTransfered('User is already in this database, dsn: {0}'.format(from_pg_dsn))

    verify_user_is_in_from_db(from_pg_dsn, user.uid)

    with transaction(from_pg_dsn) as from_pg_conn:
        PresyncPg2Pg(
            from_conn=from_pg_conn,
            to_dsn=to_pg_dsn,
            transfer_info=transfer_info,
            max_received_date=presync_received_date,
            with_presync=True,
        ).copy()
        mark_user_as_moved_from_here_in_future(to_pg_dsn, user.uid)

    log.info('Pre-sync is complete.')


def sync_user(app, user, to_db, from_db, presync_received_date):
    log.debug('Sync %r to %r', user, to_db)
    if not to_db.shard_id or not from_db.shard_id:
        raise InvalidTransferArguments('Sync requires you to specify source and destination shard_id')

    sharddb = find_sharddb(app.args)

    from_shard_id = from_db.shard_id
    to_shard_id = to_db.shard_id

    shard_id_in_sharpei = get_shard_id(uid=user.uid, dsn=sharddb)
    if shard_id_in_sharpei is None or shard_id_in_sharpei != from_shard_id:
        raise InvalidTransferArguments('Bad from_shard_id')

    if from_shard_id == to_shard_id:
        raise UserAlreadyTransfered('User is already in shard_id={0}'.format(from_shard_id))

    transfer_info = make_transfer_info(user, to_db, from_db)

    from_pg_dsn = get_connstring_by_id(app.args.sharpei, from_shard_id, app.args.maildb_dsn_suffix)
    to_pg_dsn = get_connstring_by_id(app.args.sharpei, to_shard_id, app.args.maildb_dsn_suffix)

    if from_pg_dsn == to_pg_dsn:
        raise UserAlreadyTransfered('User is already in this database, dsn: {0}'.format(from_pg_dsn))

    verify_user_is_in_from_db(from_pg_dsn, user.uid)

    with locked_transaction(from_pg_dsn, user.uid) as (from_pg_conn, from_next_revision):
        assert from_user_is_here(from_pg_conn, user.uid), 'User must be marked as is_here in source shard'
        lock_contacts_user(from_pg_conn, user.uid)
        prepare_for_migrate_subs(from_pg_conn, user.uid)
        SyncPg2Pg(
            from_conn=from_pg_conn,
            to_dsn=to_pg_dsn,
            transfer_info=transfer_info,
            min_received_date=presync_received_date,
            with_presync=True,
        ).copy()
        transfer_delete_queue(from_pg_conn, to_pg_dsn, user.uid)
        mark_user_as_moved_from_here(from_pg_conn, user.uid)
        mark_contacts_user_as_moved_from_here(from_pg_conn, user.uid, False)
        maildb_write_transfer_info(from_pg_conn, transfer_info, user.uid)
        update_subs_status(to_pg_dsn, user.uid)
        update_shard_id(sharddb, user.uid, to_db.shard_id, False)
    try:
        write_transfer_info(find_huskydb(app.args), user.uid, transfer_info)
    except Exception as e:  # pylint: disable=W0703
        log.exception('write_transfer_info to huskydb failed: %r', e)
    log.info('Sync is complete.')
