# coding: utf-8

from future.standard_library import install_aliases
install_aliases()

from time import sleep
from contextlib import contextmanager, closing
from urllib.request import urlopen, Request
from urllib.error import HTTPError, URLError
from urllib.parse import urlencode
from io import BytesIO as SIO
import socket

import logging

from six import text_type

log = logging.getLogger(__name__)

BASE_TIMEOUT = 3


def url_join(host, port=None, method=None, args=None):
    url = [host]
    if port is not None:
        url += [':', str(port)]
    if method is not None:
        if not host.endswith('/'):
            url.append('/')
        url.append(method)
    if args:
        url += ['?', urlencode(args)]
    return ''.join(url)


__RETRY_PROGRESSION = (0.1, 0.3, 1, None)


class HTTPErrorWithBody(HTTPError):
    def __init__(self, exc, body):
        super(HTTPErrorWithBody, self).__init__(exc.url, exc.code, exc.msg, exc.hdrs, SIO(body))
        self.body = body


@contextmanager
def request(url, data=None, timeout=BASE_TIMEOUT, do_retries=False,
            headers=None, log_post_data=True, skip_retry_codes=None, **urlopen_kwargs):
    if data and log_post_data:
        log.info('HTTP: %s timeout: %r data:<%r>', url, timeout, data)
    else:
        log.info('HTTP: %s timeout: %r', url, timeout)
    for try_sleep in __RETRY_PROGRESSION:
        try:
            http_request = Request(
                url,
                data=data,
                headers=headers if headers else {}
            )
            with closing(urlopen(http_request, timeout=timeout, **urlopen_kwargs)) as fd:
                response_body = fd.read()
                # aguly hack, should remove @contextmanager,
                # or beter remove this code
                # and use `bare` requests
                yield SIO(response_body)
                return
        except HTTPError as exc:
            body = exc.read()
            if not isinstance(body, text_type):
                body = body.decode('utf-8')
            if (skip_retry_codes is None or exc.code not in skip_retry_codes) and \
                    (exc.code >= 500 or do_retries) and try_sleep is not None:
                log.warning(
                    'Got temporary error, gonna retry: url=%s, code=%s, error=%s',
                    url, exc.code, body
                )
                sleep(try_sleep * timeout)
            else:
                log.exception('Got HTTP error %s: %s', url, body)
                raise HTTPErrorWithBody(exc, body.encode('utf-8'))
        except (socket.timeout, socket.error, URLError) as exc:
            # timeouts:
            #  * in python2.6 URLError.reason - socket.timeout
            #  * in python2.7 socket.timeout
            # retiries on socket.errors, eg:
            #   (104, 'Connection reset by peer')
            # URLError: <urlopen error [Errno -2] Name or service not known>:
            #  * in python2.7 URLError.reason - socket.gaierror
            is_transport_error = \
                isinstance(exc, (socket.timeout, socket.error)) or \
                isinstance(exc.reason, socket.timeout) or \
                isinstance(exc.reason, socket.gaierror)
            if is_transport_error and do_retries and try_sleep is not None:
                log.warning(
                    'Got timeout or transport error '
                    'gonna retry url=%s, reason=%s',
                    url, exc
                )
                sleep(try_sleep * timeout)
            else:
                raise
        except Exception:
            log.exception('Got unexpeted exception on %s', url)
            raise
