# coding: utf-8

from collections import namedtuple

Recipient = namedtuple('Recipient', ('name', 'email'))


def quote(name):
    """ Quote recipients if no quotes at first and last symbols

    >>> quote(u'NoQuotesAtAll')
    u'"NoQuotesAtAll"'
    >>> quote(u'"HeadingQuote')
    u'"HeadingQuote'
    >>> quote(u'"QuotedName"')
    u'"QuotedName"'
    >>> quote(u'Quotes "Inside" Name')
    u'"Quotes "Inside" Name"'

    """
    if name and not name.startswith(u'"') and not name.endswith(u'"'):
        return u'"%s"' % name
    return name


def serialize_recipients(name_email_range):
    def generate():
        for name, email in name_email_range:
            if name and email:
                yield u'%s <%s>' % (name, email)
            elif name:
                yield u'%s' % (name)
            elif email:
                yield u'<%s>' % (email)
            else:
                yield u''

    return u', '.join(generate())


if __name__ == '__main__':
    import doctest
    doctest.testmod()
