import logging
from collections import namedtuple
from datetime import datetime, timedelta

from ora2pg import copy_user
from ora2pg.cleanup import clean_user
from ora2pg.huskydb import write_transfer_info
from ora2pg.pg_put import write_transfer_info as maildb_write_transfer_info
from ora2pg.sharpei import remove_user_from_sharpei, \
    get_shard_id, get_shard_info, update_shard_id, get_connstring_by_id
from ora2pg.tools.find_master_helpers import find_sharddb, find_huskydb
from ora2pg.tools.tabs import make_tabs_list, create_tabs
from ora2pg.transfer_data import get_transfer_info, DbEndpoint, get_user_in_endpoint
from ora2pg.transfer_delete_queue import transfer_delete_queue
from ora2pg.transfer_subscriptions import prepare_for_migrate_subs, update_subs_status
from mail.pypg.pypg.common import transaction
from pymdb.common import locked_transaction
from pymdb.tools import mark_user_as_moved_from_here, lock_contacts_user, \
    mark_contacts_user_as_moved_from_here

log = logging.getLogger(__name__)


class TransferError(Exception):
    pass


class InvalidTransferArguments(TransferError):
    pass


class UserNotInFromDb(TransferError):
    pass


class TransferFailedDbChanged(TransferError):
    pass


class UserAlreadyTransfered(TransferError):
    pass


class UserBlocked(TransferError):
    pass


def check_user_is_blocked_by_concurrent_operation(conn, uid):
    cur = conn.cursor()
    cur.execute(
        '''
        select 1
          from mail.archives
         where uid = %(uid)s
           and state != 'archivation_complete'
        ''',
        dict(uid=uid),
    )
    if cur.rowcount > 0:
        raise UserBlocked('Cant transfer user now due to an active concurrent operation')


def verify_from_db(from_db, transfer_info):
    if from_db is None:
        return
    src_db_endpoint = transfer_info.src.db
    if str(from_db) != str(src_db_endpoint):
        raise UserNotInFromDb('Expected: %r, but really in: %r'
                              % (from_db, src_db_endpoint))


class FirstlineOptions(object):
    DEFAULT_TARGET_LEN = 120
    DEFAULT_MAX_DATE = datetime.now() - timedelta(days=365)

    def __init__(self, cut_fl, target_len, max_date):
        self.cut_fl = True if cut_fl is None else cut_fl
        self.target_len = target_len or FirstlineOptions.DEFAULT_TARGET_LEN
        self.max_date = datetime.fromisoformat(max_date) if max_date else self.DEFAULT_MAX_DATE


class TransferOptions(object):
    def __init__(self, fill_change_log, tabs_mapping=None, force=False, firstline_options=None, min_received_date=None):
        self.fill_change_log = fill_change_log
        self.tabs_mapping = tabs_mapping
        self.force = force
        self.min_received_date = min_received_date

        if firstline_options is None:
            self.firstline_options = FirstlineOptions(None, None, None)
        else:
            self.firstline_options = FirstlineOptions(
                firstline_options.get('use'),
                firstline_options.get('target_len'),
                firstline_options.get('max_date'),
            )


TransferArtefacts = namedtuple(
    'TransferArtefacts',
    ['info', 'mapper', 'converted_reminders'])


def from_user_is_here(conn, uid):
    cur = conn.cursor()
    cur.execute('SELECT is_here from mail.users where uid = %s', (uid,))
    if cur.rowcount == 0:
        return False
    return cur.fetchone()[0]


def verify_user_is_in_from_db(from_dsn, uid):
    with transaction(from_dsn) as conn:
        if not from_user_is_here(conn, uid):
            raise UserNotInFromDb('User absent in source Db')


class Transfer(object):

    def __init__(self, app):
        self.app = app
        self._sharddb = None
        self._huskydb = None

    @property
    def args(self):
        return self.app.args

    @property
    def sharddb(self):
        if self._sharddb is None:
            self._sharddb = find_sharddb(self.args)
        return self._sharddb

    @property
    def huskydb(self):
        if self._huskydb is None:
            self._huskydb = find_huskydb(self.args)
        return self._huskydb

    def remove_user_from_sharpei_ignore_errors(self, user):
        try:
            remove_user_from_sharpei(self.sharddb, user.uid)
        except Exception as exc:  # pylint: disable=W0703
            log.warning(
                'Got %s while try remove user from sharpei',
                exc
            )

    def pg2pg(self, user, from_db, to_db, options):
        if not to_db.shard_id:
            raise InvalidTransferArguments(
                'Transfer between shard requires you to '
                'specify destination shard_id')

        if options.force:
            shard_info = get_shard_info(uid=user.uid, dsn=self.sharddb)
            from_shard_id, is_deleted = shard_info.shard_id, shard_info.is_deleted
        else:
            from_shard_id, is_deleted = get_shard_id(uid=user.uid, dsn=self.sharddb), False

        to_shard_id = to_db.shard_id
        if from_shard_id == to_shard_id:
            raise UserAlreadyTransfered(
                'User is already in shard_id={0}'.format(from_shard_id))
        transfer_info = get_transfer_info(
            src=get_user_in_endpoint(
                db_endpoint=DbEndpoint.make_pg(from_shard_id),
                uid=user.uid,
            ),
            dst=get_user_in_endpoint(
                db_endpoint=to_db,
                uid=user.uid,
            ),
        )
        verify_from_db(from_db, transfer_info)
        from_pg_dsn = get_connstring_by_id(
            self.args.sharpei,
            from_shard_id,
            self.args.maildb_dsn_suffix
        )
        to_pg_dsn = get_connstring_by_id(
            self.args.sharpei,
            to_shard_id,
            self.args.maildb_dsn_suffix
        )
        if from_pg_dsn == to_pg_dsn:
            raise UserAlreadyTransfered(
                'User is already in this database, dsn: {0}'.format(
                    from_pg_dsn))
        verify_user_is_in_from_db(from_pg_dsn, user.uid)

        with locked_transaction(from_pg_dsn, user.uid) as (from_pg_conn, from_next_revision):
            assert from_user_is_here(from_pg_conn, user.uid), 'User must be marked as is_here in source shard'
            check_user_is_blocked_by_concurrent_operation(from_pg_conn, user.uid)
            clean_user(user.uid, to_pg_dsn)
            lock_contacts_user(from_pg_conn, user.uid)
            prepare_for_migrate_subs(from_pg_conn, user.uid)
            create_tabs(from_pg_conn, user.uid, make_tabs_list(options.tabs_mapping))
            copy_user.pg2pg(
                from_pg_conn,
                to_pg_dsn,
                transfer_info,
                options.fill_change_log,
                options.tabs_mapping,
                options.firstline_options,
                options.min_received_date,
            )
            transfer_delete_queue(from_pg_conn, to_pg_dsn, user.uid)
            mark_user_as_moved_from_here(from_pg_conn, user.uid)
            mark_contacts_user_as_moved_from_here(from_pg_conn, user.uid, is_deleted)
            maildb_write_transfer_info(from_pg_conn, transfer_info, user.uid)
            update_subs_status(to_pg_dsn, user.uid)
            update_shard_id(self.sharddb, user.uid, to_db.shard_id, is_deleted)
            return TransferArtefacts(transfer_info, None, None)

    def transfer(self, user, from_db, to_db, options):
        log.debug('transfer %r to %r', user, to_db)
        transfer_artefacts = self.pg2pg(
            user=user,
            from_db=from_db,
            to_db=to_db,
            options=options
        )

        def write_transfer_info_job():
            write_transfer_info(
                self.huskydb,
                user.uid,
                transfer_artefacts.info)

        post_transfer_jobs = [
            write_transfer_info_job,
        ]

        for job in post_transfer_jobs:
            try:
                job()
            except Exception as e:  # pylint: disable=W0703
                log.exception('Non-essentials post-transfer job %s failed: %r',
                              job.__name__, e)
        log.info('Transfer is complete.')


def transfer(  # pylint: disable=R0913
        app, user, to_db, from_db=None,
        fill_change_log=False,
        change_tabs=True,
        force=False,
        firstline_options=None,
        min_received_date=None,
):
    Transfer(app).transfer(
        user=user,
        from_db=from_db,
        to_db=to_db,
        options=TransferOptions(
            fill_change_log=fill_change_log,
            tabs_mapping=app.args.tabs_mapping if change_tabs else None,
            force=force,
            firstline_options=firstline_options,
            min_received_date=min_received_date,
        )
    )
