from collections import namedtuple

from six import text_type, binary_type


class DbEndpoint(object):
    class IllFormed(RuntimeError):
        pass

    __slots__ = ('conn_info')

    def __init__(self, arg):
        if isinstance(arg, int):
            arg = text_type(arg)
        elif isinstance(arg, binary_type):
            arg = text_type(arg, 'utf-8')

        # Deprecated
        if arg.startswith('postgre:'):
            _, arg = arg.split('postgre:')
        self.conn_info = arg

    def __eq__(self, other):
        return self.conn_info == other.conn_info

    @property
    def dsn(self):
        return self.conn_info

    @property
    def db(self):
        return self.conn_info

    @property
    def shard_id(self):
        if not self.conn_info:
            return None
        try:
            return int(self.conn_info)
        except TypeError as e:
            raise RuntimeError(
                'Given conn info %s can\'t be interpreted as shard_id: %r',
                self.conn_info,
                e
            )

    @property
    def postgre(self):
        return True

    @classmethod
    def make_pg(cls, shard_id):
        if isinstance(shard_id, int):
            shard_id = text_type(shard_id)
        elif isinstance(shard_id, binary_type):
            shard_id = text_type(shard_id, 'utf-8')
        return cls(shard_id)

    def __str__(self):
        return self.db

    def __repr__(self):
        return '%s(%s)' % (self.__class__.__name__, str(self))


UserInEndpoint = namedtuple(
    'UserInEndpoint',
    ('db', 'uid')
)


def get_user_in_endpoint(db_endpoint, uid):
    return UserInEndpoint(
        db=db_endpoint,
        uid=int(uid),
    )


TransferInfo = namedtuple(
    'TransferInfo',
    ('src', 'dst', 'started', 'script', 'script_revision', 'transfered_at')
)


def get_transfer_info(src, dst):
    import socket
    from sys import argv
    from datetime import datetime
    from ora2pg.app.code_version import get_code_version

    return TransferInfo(
        src=src,
        dst=dst,
        started=datetime.now(),
        script=argv[0],
        script_revision=get_code_version(),
        transfered_at=socket.gethostname(),
    )
