import logging
from time import sleep

from mail.pypg.pypg.common import transaction, qexec
from mail.pypg.pypg.query_conf import load_from_package

from pymdb.common import Q as COMMON_Q
from pymdb.queries import Queries
from pymdb.types.db_enums.subscription_state import SubscriptionState

log = logging.getLogger(__name__)

QUERIES = load_from_package(__package__, __file__)
RETRY_PROGRESSION = (2, 3, 5, 20, 30, 60, 180, 300)


class TransferSubscriptionError(Exception):
    pass


class BadSubscriptionError(TransferSubscriptionError):
    pass


class CantWait(TransferSubscriptionError):
    pass


class Sleeper(object):
    def __init__(self, retry_progression):
        self.try_sleep_iter = iter(retry_progression)

    def sleep(self):
        try:
            sleep(next(self.try_sleep_iter))
        except StopIteration:
            raise CantWait("Can't wait for subscriptions, try later")


def get_shared_folder_subscriptions(conn, uid):
    q = Queries(conn, uid)
    return q.shared_folder_subscriptions()


class _SubscriptionsAwaiter(object):
    def __init__(self, conn, uid, retry_progression):
        self.conn = conn
        self.uid = uid
        self.sleeper = Sleeper(retry_progression)

    def _check_subs_states(self, subscriptions, valid_states):
        for sub in subscriptions:
            if sub.state not in valid_states:
                raise BadSubscriptionError("Can't transfer subscription {0} in bad state: {1}".format(
                    sub.subscription_id, sub.state))

    def wait_for_subs(self, waiting_states, ready_states):
        is_waiting = True
        valid_states = waiting_states + ready_states
        while is_waiting:
            is_waiting = False
            subscriptions = get_shared_folder_subscriptions(self.conn, self.uid)
            self._check_subs_states(subscriptions, valid_states)
            for sub in subscriptions:
                if sub.state in waiting_states:
                    is_waiting = True
                    self.sleeper.sleep()
                    break


def wait_for_normal_working_subs(conn, uid, retry_progression=RETRY_PROGRESSION):
    sub_awaiter = _SubscriptionsAwaiter(conn, uid, retry_progression)
    waiting_states = (SubscriptionState.new, SubscriptionState.init,
                      SubscriptionState.discontinued, SubscriptionState.clear)
    ready_states = (SubscriptionState.sync, SubscriptionState.terminated,
                    SubscriptionState.migrate, SubscriptionState.migrate_finished)
    sub_awaiter.wait_for_subs(waiting_states, ready_states)


def wait_for_migrate_finish_subs(conn, uid, retry_progression=RETRY_PROGRESSION):
    sub_awaiter = _SubscriptionsAwaiter(conn, uid, retry_progression)
    waiting_states = (SubscriptionState.migrate, )
    ready_states = (SubscriptionState.terminated, SubscriptionState.migrate_finished)
    sub_awaiter.wait_for_subs(waiting_states, ready_states)


def set_migrate_for_subs(conn, uid):
    qexec(conn, QUERIES.set_migrate, uid=uid)
    conn.commit()
    qexec(conn, COMMON_Q.lock_user, uid=uid)


def set_sync_for_subs(dsn, uid):
    with transaction(dsn) as conn:
        qexec(conn, QUERIES.set_sync, uid=uid)


def update_subs_status(dsn, uid):
    with transaction(dsn) as conn:
        qexec(conn, QUERIES.update_subs_status, uid=uid)


def prepare_for_migrate_subs(conn, uid):
    wait_for_normal_working_subs(conn, uid)
    set_migrate_for_subs(conn, uid)
    wait_for_migrate_finish_subs(conn, uid)
