#include <iostream>
#include <sstream>

#include <ymod_webserver/server.h>
#include <ymod_webserver/response.h>
#include <ymod_webserver/request.h>

#include <yplatform/find.h>
#include <yplatform/module_registration.h>
#include <yplatform/loader.h>

#include <ymod_tvm/tvm.h>

#include <tvm_guard/module.h>
#include <tvm_guard/log_tskv.h>
#include <tvm_guard/log_simple.h>
#include <tvm_guard/tvm_guard.h>

#include <mail/tvm_guard/ymod_tvm/ymod_tvm.h>

#include <logdog/backend/yplatform_log.h>
#include <logdog/attributes/mail_attributes.h>

#include <boost/algorithm/string/case_conv.hpp>

namespace example {

constexpr static auto formatter = ::logdog::tskv::make_formatter(BOOST_HANA_STRING("mail-example-tskv-log"));

inline auto getModuleLogger() {
    return ::logdog::make_log(
        formatter,
        std::make_shared<yplatform::log::source>(YGLOBAL_LOG_SERVICE, "example")
    );
}

using ModuleLogger = decltype(getModuleLogger());

using TvmGuard = tvm_guard::Guard<ymod_tvm::tvm2_module>;

struct Request {
    ymod_webserver::response_ptr stream;

    std::optional<std::string> getOptionalArg(const std::string& arg) const {
        if (auto it = stream->request()->url.params.find(arg); it != stream->request()->url.params.end()) {
            return std::make_optional(it->second);
        } else {
            return std::nullopt;
        }
    }

    std::optional<std::string> getOptionalHeader(std::string arg) const {
        boost::algorithm::to_lower(arg);
        if (auto it = stream->request()->headers.find(arg); it != stream->request()->headers.end()) {
            return std::make_optional(it->second);
        } else {
            return std::nullopt;
        }
    }

    std::string endpoint() const {
        return stream->request()->url.make_full_path();
    }
};

struct Server: public yplatform::module {
    void init(const yplatform::ptree& cfg) {
        auto serverModule = yplatform::find<ymod_webserver::server>("webserver");
        auto tvmModule = yplatform::find<ymod_tvm::tvm2_module, std::shared_ptr>("ymod_tvm");
        auto dummyGuardModule = yplatform::find<tvm_guard::Module>("dummy");
        auto yModTvmGuardModule = yplatform::find<tvm_guard::Module>("with_ymod_tvm");

        const auto tvmGuard = std::make_shared<TvmGuard>(
            tvm_guard::init(
                cfg.get_child("tvm_guard"), tvmModule
            )
        );

        dummyGuardModule->bind(*serverModule, "", {"/dummy_handler"}, [](ymod_webserver::response_ptr s) {
            s->result(ymod_webserver::codes::ok, "ok");
        });

        yModTvmGuardModule->bind(*serverModule, "", {"/usual_handler"}, [](ymod_webserver::response_ptr s) {
            s->result(ymod_webserver::codes::ok, "ok");
        });

        yModTvmGuardModule->bind(*serverModule, "", {"/usual_handler_with_additional_tvm_info"},
                                 [](ymod_webserver::response_ptr s, tvm_guard::Response tvmResp) {
            s->result(ymod_webserver::codes::ok, "ok for uid: " + tvmResp.issuerUid.get_value_or(""));
        });

        serverModule->bind("", {"/custom_logic_handler"}, [tvmGuard](ymod_webserver::response_ptr s) {
            const ymod_webserver::request& req = *s->request();

            std::string uid;
            std::optional<std::string> service;
            std::optional<std::string> user;
            std::string reqId;
            std::string method = req.url.make_full_path();

            if (auto it = req.url.params.find("uid"); it != req.url.params.end()) {
                uid = it->second;
            }
            if (auto it = req.headers.find("x-ya-service-ticket"); it != req.headers.end()) {
                service = it->second;
            }
            if (auto it = req.headers.find("x-ya-user-ticket"); it != req.headers.end()) {
                user = it->second;
            }
            if (auto it = req.headers.find("x-request-id"); it != req.headers.end()) {
                reqId = it->second;
            }

            tvm_guard::Response guardResponse = tvmGuard->check(method, uid, service, user);

            LOGDOG_(getModuleLogger(), notice, logdog::response=guardResponse);

            if (guardResponse.action == tvm_guard::Action::reject) {
                s->result(ymod_webserver::codes::unauthorized, "rejected by tvm");
            } else {
                s->result(ymod_webserver::codes::ok, "ok");
            }
        });

        serverModule->bind("", {"/custom_logic_handler2"}, [tvmGuard](ymod_webserver::response_ptr s) {
            Request req{s};

            tvm_guard::Response guardResponse = tvmGuard->check(req);

            std::ostringstream out;
            out << guardResponse;
            LOGDOG_(getModuleLogger(), notice, logdog::message=out.str());

            if (guardResponse.action == tvm_guard::Action::reject) {
                s->result(ymod_webserver::codes::unauthorized, "rejected by tvm");
            } else {
                s->result(ymod_webserver::codes::ok, "ok");
            }
        });

        LOGDOG_(getModuleLogger(), notice, logdog::module=*tvmGuard);
    }
};

}

DEFINE_SERVICE_OBJECT(example::Server)


int main(int argc, char* argv[]) {
    if (argc != 2) {
        std::cout << "usage " << argv[0] << " <config>\n";
        return 1;
    }

    return yplatform_start(argv[1]);
}
