#pragma once

#include <ostream>
#include <tvm_guard/types.h>
#include <tvm_guard/guard.h>
#include <tvm_guard/log.h>
#include <boost/algorithm/string/join.hpp>

namespace tvm_guard {

inline std::ostream& operator<<(std::ostream& out, Reason reason) {
    out << toString(reason);
    return out;
}

inline std::ostream& operator<<(std::ostream& out, Action action) {
    out << (action == Action::accept ? "accept" : "reject");
    return out;
}

template<class Tvm2Module>
inline std::ostream& Guard<Tvm2Module>::print(std::ostream& out) const {
    const std::string emptyBoostOptionalValue = "none";

    out << std::boolalpha
        << "default_action=" << defaultAction << "\n"
        << "bb_env=" << (bbEnv ? std::to_string(static_cast<int>(*bbEnv)) : std::string("none")) << "\n"
        << "root_client_id=" << (rootClientId ? std::to_string(*rootClientId) : emptyBoostOptionalValue) << "\n"
        << "strong_uid_check=" << strongUidCheck << "\n"
        << "rules: " << "\n";

    for (const tvm_guard::Rule& rule : rules) {
        out << "paths=" << boost::algorithm::join(rule.paths, ",") << " "
            << "default_action=" << rule.defaultAction << " "
            << "accept_by_service=" << join(rule.acceptByService) << " "
            << "accept_by_user=" << join(rule.acceptByUser) << " "
            << "\n";
    }

    return out;
}

template<class Tvm2Module>
inline std::ostream& operator<<(std::ostream& out, const Guard<Tvm2Module>& guard) {
    return guard.print(out);
}

inline std::ostream& operator<<(std::ostream& out, const Response& response) {
    const std::string emptyBoostOptionalValue = "none";

    out << "action="  << response.action
        << " reason=" << response.reason
        << " source=" << (response.source ? std::to_string(*response.source)
                                          : emptyBoostOptionalValue)
        << " uids_from_user_ticket=" << (response.uidsFromUserTicket ? boost::algorithm::join(*response.uidsFromUserTicket, ",")
                                                                     : emptyBoostOptionalValue)

        << " issuer_uid=" << (response.issuerUid.get_value_or(emptyBoostOptionalValue))
        << " error_category=" << response.error.category().name()
        << " error_value=" << response.error.value()
        << " error_message=" << response.error.message();

    return out;
}

} // namespace tvm_guard
