#pragma once

#include <yplatform/tskv/tskv.h>
#include <tvm_guard/log_simple.h>
#include <tvm_guard/types.h>
#include <tvm_guard/guard.h>
#include <logdog/logger.h>
#include <logdog/backend/yplatform_log.h>
#include <logdog/format/tskv.h>


namespace tvm_guard {

template <typename Out>
inline void to_tskv(Out& out, const char* key, Action action) {
    out << ytskv::attr(key, (action == Action::accept ? "accept" : "reject"));
}

template <typename Out>
inline void to_tskv(Out& out, const char* key, Reason reason) {
    out << ytskv::attr(key, toString(reason));
}

template <typename Out>
inline void to_tskv(Out& out, const char* key, const Response& response) {
    boost::optional<std::string> uidsFromUserTicket;
    if (response.uidsFromUserTicket) {
        uidsFromUserTicket = boost::algorithm::join(*response.uidsFromUserTicket, ",");
    }

    const std::string name = key;
    const std::string actionName = name+".action";
    const std::string reasonName = name+".reason";
    const std::string errorCategoryName = name+".error.category";
    const std::string errorValueName = name+".error.value";
    const std::string errorMessageName = name+".error.message";

    to_tskv(out, actionName.c_str(), response.action);
    to_tskv(out, reasonName.c_str(), response.reason);

    out << ytskv::attr(name+".source", response.source)
        << ytskv::attr(name+".uids_from_user_ticket", uidsFromUserTicket)
        << ytskv::attr(name+".issuer_uid", response.issuerUid)
        << ytskv::attr(errorCategoryName.c_str(), response.error.category().name())
        << ytskv::attr(errorValueName.c_str(), response.error.value())
        << ytskv::attr(errorMessageName.c_str(), response.error.message());
}

template <typename T>
template <typename Out>
inline void Guard<T>::to_tskv(Out& out, const char* key) const {
    const std::string name = key;
    const std::string default_action = name+".default_action";
    const std::string bb_env = name+".bb_env";
    const std::string root_client_id = name+".root_client_id";
    const std::string strong_uid_check = name+".strong_uid_check";

    const std::string end = " ";
    const std::string separator = ":";

    tvm_guard::to_tskv(out, default_action.c_str(), defaultAction);

    out << (bbEnv
            ? ytskv::attr(bb_env, std::to_string(static_cast<int>(*bbEnv)))
            : ytskv::attr(bb_env, std::string("none")))
        << ytskv::attr(root_client_id, rootClientId)
        << ytskv::attr(strong_uid_check, strongUidCheck);

    for (std::size_t i = 0; i < rules.size(); i++) {
        const tvm_guard::Rule& rule = rules[i];

        const std::string key = name + ".rules." + std::to_string(i);

        const std::string paths = "paths";
        const std::string rule_default_action = "rule_default_action";
        const std::string accept_by_service = "accept_by_service";
        const std::string accept_by_user = "accept_by_user";

        std::ostringstream sout;
        sout << "(" << std::boolalpha
             << paths << separator << boost::algorithm::join(rule.paths, ",") << end
             << rule_default_action << separator << rule.defaultAction << end
             << accept_by_service << separator << join(rule.acceptByService) << end
             << accept_by_user << separator << join(rule.acceptByUser) << end
             << ")";

        out << ytskv::attr(key, sout.str());
    }
}

template <typename Out, typename Tvm2Module>
inline void to_tskv(Out& out, const char* key, const Guard<Tvm2Module>& guard) {
    guard.to_tskv(out, key);
}

inline auto getLogger(const std::string& name) {
    constexpr static auto formatter = ::logdog::tskv::make_formatter(BOOST_HANA_STRING("mail-tvm_guard-tskv-log"));
    auto logger = std::make_shared<yplatform::log::source>(YGLOBAL_LOG_SERVICE, name);
    return ::logdog::make_log(formatter, logger);
}

using Logger = decltype(getLogger(""));

}

namespace logdog {

namespace attr {

namespace tvm_guard {

LOGDOG_DEFINE_ATTRIBUTE(::tvm_guard::Action, action)
LOGDOG_DEFINE_ATTRIBUTE(::tvm_guard::Reason, reason)
LOGDOG_DEFINE_ATTRIBUTE(::tvm_guard::Response, response)

}

using namespace tvm_guard;

}

using namespace attr;

}
