#pragma once

#include <boost/range/adaptor/transformed.hpp>
#include <boost/range/algorithm/copy.hpp>
#include <boost/property_tree/ptree.hpp>
#include <library/cpp/tvmauth/checked_user_ticket.h>
#include <tvm_guard/types.h>
#include <tvm_guard/guard.h>
#include <tuple>
#include <unordered_set>

namespace tvm_guard {

inline Action fromString(const std::string& value, const std::string& name) {
    if (value == "accept") {
        return Action::accept;
    } else if (value == "reject") {
        return Action::reject;
    }

    throw parse_error::ActionParse(value, name);
}

template<class BbEnvType>
inline bool blackbox_env_by_name(const std::string& name, BbEnvType& env) {
    static const std::map<std::string, std::pair<NTvmAuth::EBlackboxEnv, TA_EBlackboxEnv>> envsByName = {
        { "blackbox",           { NTvmAuth::EBlackboxEnv::Prod,       TA_BE_PROD }},
        { "blackbox-test",      { NTvmAuth::EBlackboxEnv::Test,       TA_BE_TEST }},
        { "blackbox-corp",      { NTvmAuth::EBlackboxEnv::ProdYateam, TA_BE_PROD_YATEAM }},
        { "blackbox-corp-test", { NTvmAuth::EBlackboxEnv::TestYateam, TA_BE_TEST_YATEAM }},
        { "blackbox-stress",    { NTvmAuth::EBlackboxEnv::Stress,     TA_BE_STRESS }},
        { "blackbox-mimino",    { NTvmAuth::EBlackboxEnv::Prod,       TA_BE_PROD }}
    };

    auto it = envsByName.find(name);
    if (it == envsByName.end()) {
        return false;
    } else {
        env = std::get<BbEnvType>(it->second);
        return true;
    }
}
template<class BbEnvType, class OptPtree>
boost::optional<BbEnvType> parseBlackboxEnv(const OptPtree& env) {
    boost::optional<BbEnvType> bbEnv;
    if (env) {
        bbEnv = BbEnvType();
        if (!blackbox_env_by_name(*env, *bbEnv)) {
            throw parse_error::BbEnv(*env);
        }
    }

    return bbEnv;
}

template<class TvmModule>
inline Guard<TvmModule> init(const boost::property_tree::ptree& node, std::shared_ptr<TvmModule> module) {
    const Action defaultAction = fromString(node.get("default", ""), "default");

    boost::optional<ClientId> rootClientId = boost::make_optional(false, 0u);
    try {
        if (node.count("root_client_id")) {
            rootClientId = node.get<u_int32_t>("root_client_id");
        }
    } catch(const std::exception& ex) {
        throw parse_error::RootClientId(ex.what());
    }

    bool strongUidCheck;
    try {
        strongUidCheck = node.get<bool>("strong_uid_check");
    } catch(const std::exception& ex) {
        throw parse_error::StrongUidCheck(ex.what());
    }

    boost::optional<TA_EBlackboxEnv> bbEnv = parseBlackboxEnv<TA_EBlackboxEnv>(node.get_optional<std::string>("bb_env"));

    std::map<std::string, ClientId> clients;
    auto confClients = node.equal_range("clients");
    for (auto it = confClients.first; it != confClients.second; ++it) {
        const std::string name = it->second.get<std::string>("name");
        const ClientId id = it->second.get<ClientId>("id");
        clients[name] = id;
    }

    const auto getStringVector = [&] (const auto& range) {
        std::vector<std::string> ret;
        for (auto it = range.first; it != range.second; ++it) {
            ret.push_back(it->second.template get_value<std::string>());
        }
        return ret;
    };

    std::vector<Rule> rules;
    auto confRules = node.equal_range("rules");
    for (auto it = confRules.first; it != confRules.second; ++it) {
        Rule rule;
        const boost::property_tree::ptree& node = it->second;
        const std::string name = it->second.get<std::string>("name");
        rule.defaultAction = fromString(node.get<std::string>("default_action"), name + " rule_default_action");
        rule.paths = getStringVector(node.equal_range("paths"));

        const auto find = [&name, &clients](const std::string& s) {
            const auto it = clients.find(s);
            if (it == clients.end()) {
                throw parse_error::MissingClient(name, s);
            }
            return it->second;
        };

        boost::copy(getStringVector(node.equal_range("accept_by_service"))
                    | boost::adaptors::transformed(find),
                    std::inserter(rule.acceptByService, rule.acceptByService.begin()));

        boost::copy(getStringVector(node.equal_range("accept_by_user"))
                    | boost::adaptors::transformed(find),
                    std::inserter(rule.acceptByUser, rule.acceptByUser.begin()));

        if (!bbEnv && !rule.acceptByUser.empty()) {
            throw parse_error::EmptyBbEnvButCheckByUserIsEnabled(name);
        }

        std::set<ClientId> intersection;
        std::set_intersection(rule.acceptByService.begin(), rule.acceptByService.end(),
                              rule.acceptByUser.begin(), rule.acceptByUser.end(),
                              std::inserter(intersection, intersection.begin()));

        if (!intersection.empty()) {
            std::ostringstream out;
            boost::copy(intersection
                        | boost::adaptors::transformed([](const auto& i) { return std::to_string(i); }),
                        std::ostream_iterator<std::string>(out, ","));

            throw parse_error::ClientInServiceAndUserSection(name, out.str());
        }

        rules.emplace_back(std::move(rule));
    }

    std::unordered_set<std::string> paths;
    for (auto it = confRules.first; it != confRules.second; ++it) {
        const boost::property_tree::ptree& node = it->second;
        std::vector<std::string> confPaths = getStringVector(node.equal_range("paths"));
        
        for (auto& path: confPaths) {
            if (paths.count(path)) {
                throw parse_error::MultiplePathMention(path);
            }
            paths.insert(std::move(path));
        }
    }

    return Guard<TvmModule>(std::move(rules), defaultAction, bbEnv, rootClientId, strongUidCheck, module);
}

}
