#pragma once

#include <boost/optional.hpp>
#include <boost/system/error_code.hpp>
#include <ostream>
#include <vector>
#include <string>
#include <map>
#include <set>

namespace tvm_guard {

using ClientId = u_int32_t;

enum class Action {
    accept,
    reject
};

enum class Reason {
    rule,
    rootServiceTicket,
    wrongServiceTicket,
    wrongUserTicket,
    uidsMismatch,
    rootServiceTicketWithoutUid,
    defaultPolicy,
    ruleDefaultPolicy,
    unknownService
};

struct Response {
    Action action = Action::reject;
    Reason reason = Reason::unknownService;
    boost::system::error_code error;
    boost::optional<u_int32_t> source;
    boost::optional<std::vector<std::string>> uidsFromUserTicket;
    boost::optional<std::string> issuerUid;
    boost::optional<std::string> defaultUid;
};

class ResponseFactory {
    Response resp;

public:

    Response product() const {
        return resp;
    }

#define SETTER(name) \
    template <typename T> \
    ResponseFactory& name(const T& val) { resp.name = val; return *this; }

    SETTER(action)
    SETTER(reason)
    SETTER(error)
    SETTER(source)
    SETTER(uidsFromUserTicket)
    SETTER(defaultUid)
    SETTER(issuerUid)

#undef SETTER
};

struct Rule {
    std::vector<std::string> paths;
    Action defaultAction = Action::reject;
    std::set<ClientId> acceptByUser;
    std::set<ClientId> acceptByService;
};

struct Exception: public std::runtime_error {
    Exception(const std::string& what): std::runtime_error(what) { }
};

struct StrangeReasonError: public Exception  {
    StrangeReasonError(tvm_guard::Reason reason)
        : Exception("strange tvm_guard::Reason: " + std::to_string(static_cast<int>(reason)))
    { }
};

namespace parse_error {
struct ActionParse: public Exception {
    ActionParse(const std::string& value, const std::string& name)
        : Exception("cannot read tvm_guard::Action from string: '" + value + "' name: '" + name + "'")
    { }
};

struct StrongUidCheck: public Exception {
    StrongUidCheck(const std::string& strongUidCheck)
        : Exception("cannot parse strong_uid_check: " + strongUidCheck)
    { }
};

struct RootClientId: public Exception {
    RootClientId(const std::string& rootClientId)
        : Exception("cannot parse root_client_id: " + rootClientId)
    { }
};

struct BbEnv: public Exception {
    BbEnv(const std::string& bbEnv)
        : Exception("cannot parse 'bb_env' as TA_EBlackboxEnv: " + bbEnv)
    { }
};

struct EmptyBbEnvButCheckByUserIsEnabled: public Exception {
    EmptyBbEnvButCheckByUserIsEnabled(const std::string& ruleName)
        : Exception("cannot add 'check by user ticket' rule without 'bb_env' value. rule: " + ruleName)
    { }
};

struct MissingClient: public Exception {
    MissingClient(const std::string& rule, const std::string& client)
        : Exception("rule=" + rule + " msg='cannot find client' client=" + client)
    { }
};

struct ClientInServiceAndUserSection: public Exception {
    ClientInServiceAndUserSection(const std::string& rule, const std::string& ids)
        : Exception("rule=" + rule + " msg='client_id duplicated' ids=" + ids)
    { }
};

struct MultiplePathMention: public Exception {
    MultiplePathMention(const std::string& path)
        : Exception("msg='path is mentioned two or more times in rules' path=" + path)
    { }
};
}

namespace header {
inline std::string userTicket() {
    return "X-Ya-User-Ticket";
}

inline std::string serviceTicket() {
    return "X-Ya-Service-Ticket";
}
}

}
