#include <tvm_guard/module_impl.h>
#include <tvm_guard/log_tskv.h>
#include <tvm_guard/log.h>
#include <tvm_guard/parse.h>
#include <logdog/attributes/mail_attributes.h>

#include <yplatform/module_registration.h>
#include <ymod_webserver/codes.h>
#include <ymod_webserver/response.h>
#include <ymod_webserver/server.h>

namespace tvm_guard {

namespace log {
using namespace ::logdog::attr;
}

void rejectByTvm(ymod_webserver::response_ptr stream, const Response& tvm) {
    stream->set_code(ymod_webserver::codes::unauthorized);
    stream->set_content_type("text", "plain");
    stream->result_body(toString(tvm.reason));
}

void ModuleImpl::bind(ymod_webserver::server& server, const std::string& name,
          const std::vector<std::string>& paths, Module::Handler handler) const {
    server.bind(name, paths, [h = std::move(handler), this, m = shared_from_this()]
        (ymod_webserver::response_ptr stream) {
            const auto tvm = this->check(*stream->request());
            if (tvm.action == tvm_guard::Action::reject) {
                rejectByTvm(stream, tvm);
            } else {
                h(std::move(stream));
            }
        }
    );
}

void ModuleImpl::bind(ymod_webserver::server& server, const std::string& name,
          const std::vector<std::string>& paths, Module::Handler2 handler) const {
    server.bind(name, paths, [h = std::move(handler), this, m = shared_from_this()]
        (ymod_webserver::response_ptr stream) {
            auto tvm = this->check(*stream->request());
            if (tvm.action == tvm_guard::Action::reject) {
                rejectByTvm(stream, tvm);
            } else {
                h(std::move(stream), std::move(tvm));
            }
        }
    );
}

struct Dummy: public ModuleImpl {
    Dummy()
        : guardLogger_(getLogger(""))
    { }

    Dummy(Action defaultAction)
        : defaultAction_(defaultAction)
    { }

    virtual ~Dummy() { }

    void init(const yplatform::ptree& cfg) {
        defaultAction_ = fromString(cfg.get<std::string>("default_action"), "default_action");

        if (const auto name = cfg.get<std::string>("logger.guard", ""); !name.empty()) {
            guardLogger_ = getLogger(name);
        }

        const auto l = getLogger(cfg.get<std::string>("logger.module"));

        LOGDOG_(l, notice,
                log::action=defaultAction_,
                log::message=std::string("module initialized ") + (guardLogger_ ? "with logger.guard" : "without logger.guard"));
    }

    Response check(const ymod_webserver::request& req) const override {
        Response tvm {
            .action=defaultAction_,
            .reason=Reason::defaultPolicy
        };

        if (guardLogger_) {
            std::string reqId;
            std::string method = req.url.make_full_path();
            if (auto it = req.headers.find("x-request-id"); it != req.headers.end()) {
                reqId = it->second;
            }

            LOGDOG_(*guardLogger_, notice, log::request_id=reqId,
                    log::where_name=method, log::tvm_guard::response=tvm);
        }

        return tvm;
    }

    std::optional<Logger> guardLogger_;
    Action defaultAction_;
};

ModulePtr createDummy(Action defaultAction) {
    return std::make_shared<Dummy>(defaultAction);
}

}

DEFINE_SERVICE_OBJECT(tvm_guard::Dummy)
