#include <mail/tvm_guard/tvm_api/tvm_api.h>
#include <mail/tvm_guard/tvm_api/error.h>

namespace tvm_guard {
namespace detail {

template<>
bool TicketTraits<NTvmAuth::TCheckedServiceTicket>::ticketIsSuccessful(const NTvmAuth::TCheckedServiceTicket& ticket) {
    return ticket.GetStatus() == NTvmAuth::ETicketStatus::Ok;
}

template<>
bool TicketTraits<NTvmAuth::TCheckedUserTicket>::ticketIsSuccessful(const NTvmAuth::TCheckedUserTicket& ticket) {
    return ticket.GetStatus() == NTvmAuth::ETicketStatus::Ok;
}

template<>
std::string TicketTraits<NTvmAuth::TCheckedServiceTicket>::issuerUid(const NTvmAuth::TCheckedServiceTicket& ticket) {
    return ticket.GetIssuerUid() ? std::to_string(ticket.GetIssuerUid().GetRef())
                                 : "none";
}

template<>
boost::optional<std::string> TicketTraits<NTvmAuth::TCheckedUserTicket>::defaultUid(const NTvmAuth::TCheckedUserTicket&) {
    return boost::none;
}

template<>
bool TicketTraits<NTvmAuth::TCheckedServiceTicket>::issuerUidIsEmpty(const NTvmAuth::TCheckedServiceTicket& ticket) {
    return !static_cast<bool>(ticket.GetIssuerUid());
}

inline TString Stroka(const std::string& str) {
    return TString(str);
}

} // namespace detail

inline TString secretFromFile(const std::string& path) {
    std::ifstream fin(path);
    if (!fin.is_open()) {
        throw Exception("cannot open file with secret: " + path);
    }

    std::string secret;
    std::getline(fin, secret, '\n');

    return detail::Stroka(secret);
}

NTvmAuth::NTvmApi::TClientSettings parseSettings(const boost::property_tree::ptree& node,
                                                       std::function<TString(const std::string&)> readSecretFromFile) {
    NTvmAuth::NTvmApi::TClientSettings settings;

    try {
        settings.SetSelfTvmId(node.get<u_int32_t>("service_id"));
        settings.EnableServiceTicketChecking();
        settings.SetTvmHostPort(TString(node.get<std::string>("host")), node.get<u_int32_t>("port"));

        NTvmAuth::NTvmApi::TClientSettings::TDstMap dstMap;

        auto targetServices = node.equal_range("target_services");
        for (auto it = targetServices.first; it != targetServices.second; ++it) {
            const auto name = TString(it->second.get<std::string>("name"));
            const auto id = NTvmAuth::NTvmApi::TClientSettings::TDst(it->second.get<ClientId>("id"));
            dstMap[name] = id;
        }

        if (!dstMap.empty()) {
            TString secret = readSecretFromFile(node.get<std::string>("secret_file"));
            settings.EnableServiceTicketsFetchOptions(secret, std::move(dstMap));
        }

        auto bbEnv = parseBlackboxEnv<NTvmAuth::EBlackboxEnv>(node.get_optional<std::string>("bb_env"));
        if (bbEnv) {
            settings.EnableUserTicketChecking(*bbEnv);
        }

        settings.DiskCacheDir = node.get<TString>("disk_cache_dir", "");
    } catch(const std::exception& ex) {
        throw Exception(ex.what());
    }

    return settings;
}

boost::variant<boost::system::error_code, NTvmAuth::TCheckedServiceTicket>
TvmApiWrapper::get_native_service_ticket_or_error(const std::string& ticket) const {
    auto native = impl_->CheckServiceTicket(ticket);
    if (!native) {
        return TicketParserWrapperCategory::make_error();
    }

    return native;
}

boost::variant<boost::system::error_code, NTvmAuth::TCheckedUserTicket>
TvmApiWrapper::get_native_user_ticket_or_error(TA_EBlackboxEnv, const std::string& ticket) const {
    auto native = impl_->CheckUserTicket(ticket);

    if (!native) {
        return TicketParserWrapperCategory::make_error();
    }

    return native;
}

std::shared_ptr<TvmApiWrapper> getTvmApiWrapper(const boost::property_tree::ptree& node,
                                                const NTvmAuth::TLoggerPtr& logger) {
    NTvmAuth::NTvmApi::TClientSettings settings = parseSettings(node, secretFromFile);
    auto client = std::make_shared<NTvmAuth::TTvmClient>(settings, logger);
    return std::make_shared<TvmApiWrapper>(client);
}

}
