from yatest.common import data_path, network, source_path, output_path, work_path, build_path
from library.python import resource
from mail.webmail_config.lib.make_config import make_config
import multiprocessing
import time
import subprocess
import shlex
import sys
import os
import signal
import psutil
import re
from pathlib import Path
import yaml
from six import binary_type


STREAM_FILE_POSTFIX = '.0'
WORK_PATH_MARKER = 'work_path/'


def source_path_with_prefix(parent, name):
    return source_path(os.path.join(parent, name))


def output_path_with_prefix(parent, name):
    return output_path(os.path.join(parent, name))


def fill_file(path_src, path_dst, iterations):
    data = []
    with open(path_src, 'rb') as f:
        data = f.read()

    sleep_timeout = 1
    bytes_per_second = os.path.getsize(path_src) // iterations

    sys.stdout = open(path_dst + ".stdout", "w")
    sys.stderr = open(path_dst + ".stderr", "w")

    with open(path_dst, "wb") as f:
        for i in range(iterations):
            start = i * bytes_per_second
            finish = start + bytes_per_second
            f.write(data[start:finish])
            time.sleep(sleep_timeout)
        f.write(data[iterations * bytes_per_second:])


def find_free_port():
    with network.PortManager() as pm:
        return pm.get_port(8082)


def patch_log_file_names(config_path, file_names):
    content = None
    with open(config_path, 'r') as f:
        content = f.read()
    Path(config_path).unlink()

    content = re.sub('(' + '|'.join(map(re.escape, file_names)) + ')(?=[,}\\s]|$)', r'\1' + STREAM_FILE_POSTFIX, content, flags=re.MULTILINE)

    with open(config_path, 'w') as f:
        f.write(content)


def is_log(ext):
    ext = ext.lower()
    return ext in ('.tskv', '.log')


def is_config(ext):
    ext = ext.lower()
    return ext in ('.yml', '.yaml')


def get_write_mode(value):
    if isinstance(value, binary_type):
        return 'wb'
    return 'w'


def extract_resources_to_work_path():
    logs = []
    configs = []
    wp = work_path('')
    for key, value in resource.iteritems(WORK_PATH_MARKER):
        file_name = key.replace(WORK_PATH_MARKER, '')
        filepath = Path(os.path.join(wp, file_name))
        filepath.parent.mkdir(parents=True, exist_ok=True)

        with filepath.open(get_write_mode(value)) as f:
            f.write(value)

        ext = filepath.suffix
        if is_log(ext):
            logs.append(file_name)
        elif is_config(ext):
            configs.append(filepath)
        pass
    return configs, logs


def try_generate_config(dog_name):
    env = 'testing'
    template = resource.find('/config_template.yml')
    if not template:
        return None

    configs = []
    for _, value in resource.iteritems('/configs/'):
        configs.append(yaml.safe_load(value))

    cfg = make_config(env, template, *configs, silent=True)
    config_path = work_path(f'{dog_name}.yml')
    with open(config_path, get_write_mode(cfg)) as f:
        f.write(cfg)
    return config_path


def prepare_resources(dog_name):
    configs, logs = extract_resources_to_work_path()
    generated_cfg = try_generate_config(dog_name)
    if generated_cfg:
        configs.append(generated_cfg)

    for config in configs:
        patch_log_file_names(config, logs)
    return configs, logs


def start_unistat(script_path, unistat_cmd, file_names, port, files_is_data_files=True, iterations=60):
    def data_files_path(name):
        return data_path(name) if files_is_data_files else name

    with open("recipe.port", "w") as f:
        f.write(str(port))

    for file_name in file_names:
        Path(os.path.dirname(data_path(file_name))).mkdir(parents=True, exist_ok=True)
        Path(data_path(file_name + STREAM_FILE_POSTFIX)).touch()

    unistat_process = subprocess.Popen(shlex.split(unistat_cmd), stdout=open(data_path("cunistat.log"), "w"),
                                       stderr=open(data_path("cunistat.err"), "w"))

    with open("recipe.pid", "w") as f:
        f.write(str(unistat_process.pid))

    wait_until_log_files_open_by_unistat = 2
    time.sleep(wait_until_log_files_open_by_unistat)

    unistat_files = set()
    for file in psutil.Process(unistat_process.pid).open_files():
        unistat_files.add(file.path)

    log_fillers = []
    for name in file_names:
        path_dst = data_path(name + STREAM_FILE_POSTFIX)
        real_path = os.path.realpath(os.path.join(os.curdir, path_dst))
        if real_path not in unistat_files:
            raise RuntimeError(f'it seems that unistat does not read file: {path_dst}')

        filler = multiprocessing.Process(target=fill_file, args=(data_files_path(name), path_dst, iterations))
        log_fillers.append(filler)

    for log_filler in log_fillers:
        log_filler.start()

    for log_filler in log_fillers:
        log_filler.join()

    wait_until_unistat_finish_work = 10
    time.sleep(wait_until_unistat_finish_work)

    with open("recipe.done", "w") as f:
        f.write("done")


def find_supervisord_log(logs):
    for log in logs:
        if 'supervisord' in log.lower():
            return log
    return None


def start_dog_unistat(dog_name, unistat_log='unistat.log'):
    port = find_free_port()

    _, logs = prepare_resources(dog_name)

    script_path = source_path(f'mail/{dog_name}/unistat/cython/{dog_name}_unistat.py')
    config_path = work_path(f'{dog_name}.yml')

    unistat_cmd = f"{build_path(f'mail/{dog_name}/unistat/cython/cunistat')} {script_path} -d {work_path('')} -H '::' -p {str(port)}"

    if unistat_log:
        unistat_cmd = f'{unistat_cmd} -l {work_path(unistat_log)}'

    supervisord_log = find_supervisord_log(logs)
    if supervisord_log:
        unistat_cmd = f'{unistat_cmd} --supervisorlog {data_path(supervisord_log + STREAM_FILE_POSTFIX)}'

    unistat_cmd = f'{unistat_cmd} {config_path}'

    start_unistat(script_path, unistat_cmd, logs, port)


def start_app_unistat(argv):
    port = find_free_port()
    dog_name = resource.find('dog_name').decode('utf-8')
    unistat_log = 'unistat.log'

    _, logs = prepare_resources('config')

    script_path = source_path(f'mail/{dog_name}/unistat/cython/unistat.py')
    config_path = work_path('config.yml')

    unistat_cmd = f"{build_path(f'mail/{dog_name}/unistat/cython/cunistat')} {script_path} -d {work_path('')} -H '::' -p {str(port)} -l {work_path(unistat_log)}"

    supervisord_log = find_supervisord_log(logs)
    if supervisord_log:
        unistat_cmd = f'{unistat_cmd} --supervisorlog {data_path(supervisord_log + STREAM_FILE_POSTFIX)}'

    unistat_cmd = f'{unistat_cmd} {config_path}'

    start_unistat(script_path, unistat_cmd, logs, port)


def stop_unistat(argv):
    with open("recipe.pid") as f:
        pid = int(f.read())
    try:
        os.kill(pid, signal.SIGTERM)
    except ProcessLookupError as e:
        with open(data_path("cunistat.err")) as f:
            unistat_stderr = f.read()
            if unistat_stderr:
                raise RuntimeError(f'unistat stderr: {unistat_stderr}')
            else:
                raise e
