from argparse import ArgumentParser
from collections import namedtuple
from mail.unistat.cpp.cython.meters import (
    AccessLogCount,
    AccessLogCountByFirstStatusDigit,
    AccessLogCountByPathAndFirstStatusDigit,
    AccessLogRequestTimeHist,
    AccessLogRequestTimeHistByPath,
    HttpClientHttpRequestTotalTimeHist,
    HttpClientHttpRequestCountByStatus,
)
import os
import yaml
import json
import logging
import mail.unistat.cpp.example.cpp.run as run
import mail.unistat.cpp.cython.logs as logs


logging.basicConfig(level=logging.WARNING, format='[%(asctime)s] [%(levelname)s]: %(message)s')
log = logging.getLogger(__name__)


SERVICES = [
    ['blackbox', 'blackbox'],
    ['conninfo', 'sharpei'],
]

SERVICE_ENDPOINTS = [
    '/mark',
    '/complex_move',
]


def make_access_log_meters(name_prefix, endpoints):
    meters = [
        AccessLogCount(name_prefix),
        AccessLogCountByFirstStatusDigit(name_prefix),
        AccessLogRequestTimeHist((0, 100, 300, 1000, 5000), 'access_log_request'),
    ]

    for endpoint in endpoints:
        meters += [
            AccessLogCountByPathAndFirstStatusDigit(endpoint, name_prefix),
            AccessLogRequestTimeHistByPath((0, 100, 300, 1000, 5000), endpoint, 'access_log_request'),
        ]

    return meters


def make_http_client_log_meters(services):
    meters = [
        HttpClientHttpRequestTotalTimeHist((0, 100, 500, 1000, 5000), service[0], service[1] + "_time") for service in services
    ]
    meters += [
        HttpClientHttpRequestCountByStatus(service[0], "count_by_status_" + service[1]) for service in services
    ]

    return meters


def parse_args():
    parser = ArgumentParser()
    parser.add_argument('-H', '--host', default='::')
    parser.add_argument('-p', '--port', default=8082, type=int)
    parser.add_argument('-d', '--dir', default='.')
    parser.add_argument('-s', action='store_true', help='read file logs from start')
    parser.add_argument('-l', '--log', default='/var/log/service/unistat.log', help='path for yplatform.log')
    parser.add_argument('service_config')
    return parser.parse_args()


def make_service_config(data):
    return ServiceConfig(
        access_log=os.path.join(os.curdir, data['config']['log']['access']['sinks'][0]['path']),
        httpclient_log=os.path.join(os.curdir, data['config']['log']['http_client']['sinks'][0]['path']),
    )


ServiceConfig = namedtuple('ServiceConfig', (
    'access_log',
    'httpclient_log',
))


# example python meter
class PyHttpClientLogCount(object):
    def __init__(self, name=None, endpoint=None):
        self.__endpoint = '' if endpoint is None else endpoint
        self.__name = 'xxx' if name is None else name + endpoint
        self.__counter = 0

    def update(self, record):
        if self.__endpoint in record['host']:
            self.__counter += 1

    def get(self):
        return json.dumps([self.__name + "_summ", self.__counter], separators=(',', ':'))


def main():
    args = parse_args()
    log.info('chdir %s' % os.path.abspath(args.dir))
    os.chdir(args.dir)

    with open(args.service_config) as f:
        service_config = make_service_config(yaml.load(f, Loader=yaml.FullLoader))

    fast_forward = args.s

    access_log = logs.AccessTskv([], make_access_log_meters("access_log", SERVICE_ENDPOINTS), fast_forward, service_config.access_log)
    pm = PyHttpClientLogCount("py_http_client_log_", "sharpei")
    http_client_log = logs.HttpClientLog([pm], make_http_client_log_meters(SERVICES), fast_forward, service_config.httpclient_log)

    logs_list = [access_log, http_client_log]

    run.run(args.host, args.port, logs_list, args.log, logLevel='info')


if __name__ == '__main__':
    main()
