#pragma once

#include <mail/unistat/cpp/include/logger.h>
#include <mail/unistat/cpp/include/meter_wrappers.h>
#include <mail/unistat/cpp/include/meters/common.h>

#include <boost/asio.hpp>
#include <boost/algorithm/string/join.hpp>
#include <boost/range/adaptor/transformed.hpp>

namespace unistat {

template <template <typename...> typename Reader, typename File, typename Parser, typename Meters>
struct Log {
    Log(Log&& other) noexcept
        : _ios(std::move(other._ios))
        , _reader(std::move(other._reader))
        , _logger(std::move(other._logger))
        , _pyMeters(std::move(other._pyMeters))
        , _cppMeters(std::move(other._cppMeters))
        , _parseErrorCounter(std::move(other._parseErrorCounter))
        , _unexpectedExceptionInUpdateCounter(std::move(other._unexpectedExceptionInUpdateCounter))
    {}

    template <typename... FileArgs>
    explicit Log(std::vector<PyObject *> pyMeters, std::vector<Meters> cppMeters, bool fastForward, FileArgs... fileArgs)
        : _ios()
        , _reader(std::move(File(_ios, std::forward<FileArgs>(fileArgs)...)), fastForward)
        , _logger()
        , _pyMeters{pyMeters.begin(), pyMeters.end()}
        , _cppMeters{std::move_iterator(cppMeters.begin()), std::move_iterator(cppMeters.end())}
        , _parseErrorCounter(normalizeName(_reader.getSourceName() + "_parse_error_summ"), 0)
        , _unexpectedExceptionInUpdateCounter(normalizeName(_reader.getSourceName() + "_exception_summ"), 0)
    {}

    void parseAndUpdateMeters(std::string_view logLine) {
        try {
            const auto recordOpt = Parser::parse(logLine);

            if (recordOpt.has_value()) {
                for (auto &m : _pyMeters) {
                    m.update(*recordOpt);
                }
                for (auto &m : _cppMeters) {
                    m.update(*recordOpt);
                }
            } else {
                std::get<std::size_t>(_parseErrorCounter) += 1;
                LOGDOG_WHERE_(logger(), warning, source_name=_reader.getSourceName(), message="can't parse line",
                        line=logLine);
            }
        } catch (const std::exception& e) {
            std::get<std::size_t>(_unexpectedExceptionInUpdateCounter) += 1;
            LOGDOG_WHERE_(logger(), error, source_name=_reader.getSourceName(), line=logLine,
                    message="unexpected exception", exception=e);
        }
    }

    std::vector<std::string> get() const {
        std::vector<std::string> signals;
        for (const auto& m : _pyMeters) {
            if (std::optional<std::string> signal = m.get(); signal) {
                signals.emplace_back(std::move(*signal));
            }
        }

        for (const auto& m : _cppMeters) {
            if (std::optional<std::string> signal = m.get(); signal) {
                signals.emplace_back(std::move(*signal));
            }
        }

        signals.emplace_back(yamail::data::serialization::toJson(_parseErrorCounter).str());
        signals.emplace_back(yamail::data::serialization::toJson(_unexpectedExceptionInUpdateCounter).str());

        return signals;
    }

    std::string_view read() {
        return _reader();
    }

    void setLogger(Logger logger) {
        _reader.setLogger(logger);
        _logger = std::move(logger);
    }

private:
    Logger logger() {
        if (!_logger) {
            throw std::logic_error("Trying use uninitialized logger");
        }

        return *_logger;
    }

    boost::asio::io_service _ios;
    Reader<File> _reader;
    std::optional<Logger> _logger;
    std::vector<PyMeterWrapper> _pyMeters;
    std::vector<MeterWrapper<Meters>> _cppMeters;
    NamedValue<std::size_t> _parseErrorCounter;
    NamedValue<std::size_t> _unexpectedExceptionInUpdateCounter;
};

} // namespace unistat
