#pragma once

#include <mail/unistat/cpp/include/signal_record.h>

#include <yamail/data/serialization/json_writer.h>
#include <boost/python/refcount.hpp>
#include <Python.h>

namespace yamail::data::reflection {

template <typename R, typename P, typename Visitor>
struct apply_visitor_impl<const std::chrono::duration<R, P>, Visitor> {
    template <typename Tag>
    static void apply(Visitor& v, const std::chrono::duration<R, P> value, Tag&& tag) {
        apply_visitor(v, value.count(), std::forward<Tag>(tag));
    }
};

} // namespace yamail::data::reflection

namespace unistat {

struct GilLockGuard {
    GilLockGuard()
        : _gstate(PyGILState_Ensure())
    {}

    ~GilLockGuard() {
        PyGILState_Release(_gstate);
    }

private:
    PyGILState_STATE _gstate;
};

struct xdecref {
    void operator()(PyObject* obj) const noexcept { boost::python::xdecref(obj);}
};

using PyObjectPtr = std::unique_ptr<PyObject, xdecref>;

inline PyObjectPtr pyString(const char* str) {
#if PY_MAJOR_VERSION >= 3
    return PyObjectPtr(PyUnicode_InternFromString(str));
#else
    return PyObjectPtr(PyString_InternFromString(str));
#endif
}

inline int pySetItem(PyObject* dict, PyObjectPtr key, PyObjectPtr value) noexcept {
    if (nullptr != dict && key && value) {
        return PyDict_SetItem(dict, key.get(), value.get());
    }
    return -1;
}

inline PyObjectPtr asDict(std::string_view record) noexcept {
    auto dict = PyObjectPtr(PyDict_New());
    pySetItem(dict.get(), pyString("raw"), pyString(record.data()));
    return dict;
}

inline PyObjectPtr asDict(const std::map<std::string, std::string>& record) noexcept {
    auto dict = PyObjectPtr(PyDict_New());

    for (const auto& [k, v] : record) {
        pySetItem(dict.get(), pyString(k.data()), pyString(v.data()));
    }

    return dict;
}

inline PyObjectPtr asDict(const PaRecord& record) noexcept {
    PyObjectPtr dict(PyDict_New());

    PyObjectPtr type(PyLong_FromLong(record.type));
    pySetItem(dict.get(), pyString("type"), std::move(type));

    PyObjectPtr host(pyString(record.host));
    pySetItem(dict.get(), pyString("host"), std::move(host));

    PyObjectPtr req(pyString(record.req));
    pySetItem(dict.get(), pyString("req"), std::move(req));

    PyObjectPtr suid(pyString(record.suid));
    pySetItem(dict.get(), pyString("suid"), std::move(suid));

    PyObjectPtr spent_ms(PyLong_FromLong(record.spent_ms));
    pySetItem(dict.get(), pyString("spent_ms"), std::move(spent_ms));

    PyObjectPtr timestamp(PyLong_FromLong(record.timestamp));
    pySetItem(dict.get(), pyString("timestamp"), std::move(timestamp));

    return dict;
}

inline std::string toString(PyObject* obj) {
    std::string result;
#if PY_MAJOR_VERSION >= 3
    result = PyBytes_AS_STRING(obj);
#else
    result = PyString_AS_STRING(obj);
#endif
    return result;
}

inline void pyMethodCall(PyObject* obj, const char* methodName, PyObjectPtr params) noexcept {
    PyObjectPtr pyMethodName(pyString(methodName));

    auto res = PyObjectPtr(PyObject_CallMethodObjArgs(obj, pyMethodName.get(), params.get(), NULL));
    if (res == NULL) {
        PyErr_Print();
    }
}

inline std::optional<std::string> pyMethodCall(PyObject* obj, const char* methodName) {
    PyObjectPtr pyMethodName(pyString(methodName));

    auto res = PyObjectPtr(PyObject_CallMethodObjArgs(obj, pyMethodName.get(), NULL));
    if (res == NULL) {
        PyErr_Print();
        return std::nullopt;
    } else if (
#if PY_MAJOR_VERSION >= 3
               PyUnicode_Check(res.get())
#else
               PyString_Check(res.get())
#endif
               ) {
        return toString(res.get());
    }

    return std::nullopt;
}

template <typename T>
struct MeterWrapper {
    explicit MeterWrapper(T&& meter)
        : _meter(std::move(meter))
        , _lock()
    {}

    MeterWrapper(MeterWrapper&& other) noexcept
        : _meter(std::move(other._meter))
        , _lock()
    {}

    template <typename Record>
    void update(Record&& record) {
        std::lock_guard lock(_lock);
        std::visit([&record](auto&& meter) {
            meter->update(record);
        }, _meter);

    }

    std::optional<std::string> get() const {
        std::lock_guard lock(_lock);
        return std::visit([](const auto& m) -> std::optional<std::string> {
            return serialize(m->get());
        }, _meter);
    }

private:
    template <typename Signal>
    static decltype(auto) serialize(Signal&& v) {
        return yamail::data::serialization::toJson(v).str();
    }

    template <typename Signal>
    static std::optional<std::string> serialize(std::vector<Signal>&& v) {
        if (v.empty()) {
            return std::nullopt;
        }
        using boost::algorithm::join;
        using boost::adaptors::transformed;
        return join(v | transformed([](const auto& v){return yamail::data::serialization::toJson(v).str();}), ",");
    }

    T _meter;
    mutable std::mutex _lock;
};

template <>
struct MeterWrapper<PyObject *> {
    explicit MeterWrapper(PyObject* meter)
        : _meter(meter)
    {}

    template <typename Record>
    void update(Record&& record) {
        if (_meter) {
            GilLockGuard lock;
            pyMethodCall(_meter, "update", asDict(record));
        }
    }

    void update(std::string_view record) {
        if (_meter) {
            GilLockGuard lock;
            pyMethodCall(_meter, "update", asDict(record));
        }
    }

    std::optional<std::string> get() const {
        if (_meter) {
            GilLockGuard lock;
            return pyMethodCall(_meter, "get");
        }
        return std::nullopt;
    }

private:
    PyObject* _meter;
};

using PyMeterWrapper = MeterWrapper<PyObject*>;

} // namespace unistat
